#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "drp.h"
#include "options.h"
  
#define RMS_WEIGHT 1

#define W_TIME 0
#define W_TEFF 1
#define W_RMS  2
#define W_NONE 3

#define EPS 1.0e-6

int weigh = W_TEFF;
float wOn, wAve, wtotal;

char PROGRAM_NAME[] = "accum";
char description[] = "add work area to hold area";
char required[] = "";
struct _options options[] = {
{ "-help",		"Print out this message" },
{ "-weight",            "choose 'time', 'teff'(default), or 'none'" },
{ "-weight number",     "supply your own weight factor" },
{NULL, NULL }};

void ParseOpts(int *pargc, char ***pargv)
{
    char *opt, *optarg;

    opt = (*pargv)[0] + 1;

/* Help */
    if (!strcmp(opt, "help")) {
	Help();
	exit(EX_SUCCESS);
    }
    if (!strcmp(opt, "weight")) {
	GetOption(&optarg, pargc, pargv);
	if (!strcmp(optarg, "time")) {
	    weigh = W_TIME;
	    return;
	}
	if (!strcmp(optarg, "teff")) {
	    weigh = W_TEFF;
	    return;
	}
	if (!strcmp(optarg, "none")) {
	    weigh = W_NONE;
	    return;
	}
	if ((wOn = (float)atof(optarg)) > 0.0) {
	    weigh = W_RMS;
	    return;
	}
    }
    Syntax(**pargv);
}

SCAN OnScan, AveScan;

int main (int argc, char *argv[])
{
    int i;
    double d1, d2;

    GetOpts(argc, argv);

    GetScan (&OnScan);
    GetAve (&AveScan);

    if (AveScan.IntTime == 0) {
	if (weigh == W_RMS) {
	    OnScan.flag[9] = RMS_WEIGHT;
	    OnScan.work[9] = wOn;
	}
	PutAve (&OnScan);
    } else {
	/* consistency check */
	d1 = fabs(OnScan.RestFreq - AveScan.RestFreq);
	if (d1 > EPS) DRPwarning("disagreement in frequency (%lf)", d1);
	if (OnScan.NChannel != AveScan.NChannel) 
	  DRPerror("disagreement in number of channels");
	if (strcmp(OnScan.Name, AveScan.Name))
	  DRPwarning("disagreement in source name");
	d1 = fabs((double)(OnScan.Longitude - AveScan.Longitude));
	d2 = fabs((double)(OnScan.Latitude  - AveScan.Latitude ));
	if ((d1 > EPS) || (d2 > EPS))
	  DRPwarning("disagreement in position (%.1lf, %.1lf)", 
		     d1*RADTOSEC, d2*RADTOSEC);
	d1 = fabs((double)(OnScan.LMapOff - AveScan.LMapOff));
	d2 = fabs((double)(OnScan.BMapOff - AveScan.BMapOff));
	if ((d1 > EPS) || (d2 > EPS))
	  DRPwarning("disagreement in offset (%.1lf, %.1lf)", 
		     d1*RADTOSEC, d2*RADTOSEC);
	/* calculate weight factors: */
	switch (weigh) {
	  case W_TIME:	        /* time */
	    wOn = OnScan.IntTime;
	    wAve = AveScan.IntTime;
	    break;
	  case W_TEFF:		/* time/Tsys^2 */
	    if (OnScan.Tsys == 0.0 || AveScan.Tsys == 0.0) {
		OnScan.Tsys = 1.0;
		AveScan.Tsys = 1.0;
	    }
	    wOn = OnScan.IntTime/(OnScan.Tsys*OnScan.Tsys);
	    wAve = AveScan.IntTime/(AveScan.Tsys*AveScan.Tsys);
	    break;
	  case W_RMS:		/* rms */
	    if (AveScan.flag[9] != RMS_WEIGHT) {
		DRPerror("no weight supplied for average scan");
	    }
	    wAve = AveScan.work[9];
	    break;
	  case W_NONE:
	    wOn = 1.0;
	    wAve = 1.0;
	    break;
	}
	if (weigh != W_NONE) wtotal = wOn+wAve;
	else                 wtotal = 1.0;
  
	for (i = 0; i < OnScan.NChannel; i++) {
	    AveScan.Channels[i] =
	      (AveScan.Channels[i]*wAve+OnScan.Channels[i]*wOn)/wtotal;
	}

	switch (weigh) {
	  case W_TIME:
	    AveScan.IntTime += OnScan.IntTime;
	    break;
	  case W_TEFF:
	    AveScan.IntTime += OnScan.IntTime;
	    AveScan.Tsys = sqrt((double)AveScan.IntTime/(wOn+wAve));
	    break;
	  case W_RMS:
	    AveScan.flag[9] = RMS_WEIGHT;
	    AveScan.work[9] = wOn+wAve;
	    break;
	}
	
	PutAve (&AveScan);
    }
  
    exit(EX_SUCCESS);
}
