#include <math.h>
#include "astronomy.h"
#include "constant.h"
 
void cuv(pVector *p, cVector v)
{
   v[0] = cos(p->l)*cos(p->b);
   v[1] = sin(p->l)*cos(p->b);
   v[2] = sin(p->b);
}
 
void uvc(cVector v, pVector *p)
{
   if (v[0] == 0.0 && v[1] == 0.0) {
      p->l = 0.0;
      if (v[2] > 0.0) p->b =  PI/2.0;
      else            p->b = -PI/2.0;
   } else {
      p->l = atan2(v[1],v[0]);
      if (p->l < 0.0)  p->l += 2.0*PI;
      p->b = atan(v[2]/sqrt(v[0]*v[0]+v[1]*v[1]));
   }
}
 
void invert(rotMatrix m)
{
    double swap;
    int i, j;

    for (i = 0; i < 3; i++) {
	for (j = i+1; j < 3; j++) {
	    swap = m[j][i];
	    m[j][i] = m[i][j];
	    m[i][j] = swap;
	}
    }
}
 
void rotate(rotMatrix m, cVector vo, cVector vn)
{
    cVector v;

    v[0] = m[0][0]*vo[0]+m[0][1]*vo[1]+m[0][2]*vo[2];
    v[1] = m[1][0]*vo[0]+m[1][1]*vo[1]+m[1][2]*vo[2];
    v[2] = m[2][0]*vo[0]+m[2][1]*vo[1]+m[2][2]*vo[2];
    vn[0] = v[0];
    vn[1] = v[1];
    vn[2] = v[2];
}
 
void trc(rotMatrix m, pVector *old, pVector *new)
{
   static double vo[3], vn[3];
 
   cuv(old,vo);
   rotate(m,vo,vn);
   uvc(vn,new);
}
 
double djl(int Year, int Month, int Day)
{
   long DayNumber;
 
   DayNumber = 367L*Year - 7*(Year+(Month+9)/12)/4
	     - 3*((Year+(Month-9)/7)/100+1)/4 + 275*Month/9
	     + Day + 1721029L;
   return ((double)DayNumber-0.5);
}
 
/* 
 * Calculation of the components of the earth due to the 
 * motion of the earth around the earth moon barycenter
 * in rectangular, equatorial coordinates for
 * given elements a[0]...a[7] for the sun and the moon.
 *
 * input:
 *     a:           solar and lunar elements
 *
 * output: 
 *     vlun:        velocity vector 
 */
void vmoon(double a[], cVector vlun)
{
    double si1,si2,si3,ece,ecc;
    double tlm,pml,ain;
    double sinc,cinc,somg,comg,seps,ceps;
    double vluna,vxlun,vylun,vzlun,vxecl,vyecl,vzecl;
 
    /* the eccentricity of the lunar orbit */
    ecc = 0.054900489;
    /* equation of the center */
    si1 = sin(a[5]);
    si2 = sin(2.0*a[5]);
    si3 = sin(3.0*a[5]);
    ece = ecc*(2.0*si1+ecc*(1.25*si2+ecc*(1.083333*si3-0.25*si1)));
    /* the true lunar longitude of the moon */
    tlm = a[3]-a[4]+ece;
    /* the lunar longitude of lunar perigee */
    pml = a[3]-a[4]-a[5];
    /* the lunar velocity perpendicular to the radius vector (m/s) */
    vluna = 1023.165604/sqrt(1.0-ecc*ecc);
    /* components with respect to ascending node and pole of orbit */
    vxlun = -vluna*(sin(tlm)+ecc*sin(pml));
    vylun =  vluna*(cos(tlm)+ecc*cos(pml));
    vzlun =  0.0; 
    /* components with respect to eclipse */
    /* the inclination of the lunar orbit to the eclipse */
    ain = 0.089804108;
    sinc = sin(ain);
    cinc = cos(ain);
    /* longitude of ascending node */
    comg = cos(a[4]);
    somg = sin(a[4]);
 
    vxecl = comg*vxlun -somg*cinc*vylun +somg*sinc*vzlun; 
    vyecl = somg*vxlun +comg*cinc*vylun -comg*sinc*vzlun; 
    vzecl =                  sinc*vylun +     cinc*vzlun; 
 
    /* components with respect to equator */
    seps = sin(a[6]);
    ceps = cos(a[6]);
 
    vlun[0] = vxecl;
    vlun[1] = vyecl*ceps-vzecl*seps; 
    vlun[2] = vyecl*seps+vzecl*ceps; 
}
 
/*
 * Calculation of the components of the earth's velocity along its 
 * orbit plus the components of the observer's velocity due the ro-
 * tation of the earth in rectangular, equatorial coordinates
 * for a given local apparent sidereal time (last) in radians and elements 
 * a[0]...a[7] for the sun and the moon which result from routine 'elements'.
 * 
 * input:
 *     last:        local apparent sidereal time
 *     a:           solar and lunar elements
 *     phi:         geographical lattitude
 *     elev:        geographical elevation
 * output:
 *     vorbit:      orbital velocity of earth
 *     vrotat:      rotational velocity of earth
 *     vtel:        velocity vector of earth;
 *
 * side effects:
 *     the true longitude of the sun is calculated and stored in a[8]
 */
void vearth(double last, double rho, double phi, double a[], cVector vtel)
{
    double si1, si2, si3;
    double ece;
    double vorb, vrho;
    double sinfac, cosfac;
 
    /* equation of the center */
    si1 = sin(a[2]);
    si2 = sin(2.0*a[2]);
    si3 = sin(3.0*a[2]);
    ece = a[7]*(2.0*si1+a[7]*(1.25*si2+a[7]*(1.083333*si3-0.25*si1)));
    /* true longitude of the sun */
    a[8] = a[0]+ece;
    /* components of the earth's velocity */
    vorb = 29784.819/sqrt(1.0-a[7]*a[7]);
 
    /* vrho is the velocity due to diurnal rotation (m/s) */
    vrho = 2.0*PI*rho*ROTATE/24.0/3600.0;
    vrho = vrho*cos(phi);
 
    cosfac = cos(a[8])+a[7]*cos(a[1]);
    sinfac = sin(a[8])+a[7]*sin(a[1]);
    vtel[0] =  vorb*sinfac           - vrho*sin(last);
    vtel[1] = -vorb*cosfac*cos(a[6]) + vrho*cos(last);
    vtel[2] = -vorb*cosfac*sin(a[6]);
}
 
/*
 * Calculation of certain elements of the sun's and moon's orbit
 * input:
 *     Js:                 Julian date of epoch
 * output:
 *     a:           resulting orbital elements
 * 
 * side effects:
 *    element a[8] is reserved for true longitude of the sun
 *    which may be calculated by calling routine VEARTH 
 */
void elements(double Js, double a[])
{
    double T, ip;
    int i;
 
    T = (Js-2451545.0)/36525.0;
    /* geometric mean longitude of the sun */
    a[0] = 280.46645 + (36000.76983 + 0.0003032*T)*T;
    /* mean longitude of perigee */
    a[1] = 180.0+102.937348+(1.7195269+(0.00045962+0.000000499*T)*T)*T;
    /* mean anomaly of the sun */
    a[2] = 357.5291092+(35999.0502909-(0.0001536 + T/24490000.0)*T)*T;
    /* mean longitude of the moon */
    a[3] = 218.3164591+(481267.88134236-(0.0013268-T/545868.0+T*T/113065000.0)*T)*T;
    /* longitude of the mean ascending node */
    a[4] = 125.044555-(1934.1361849-(0.0020762+T/467410.0-T*T/60616000.0)*T)*T;
    /* mean anomaly of the moon */
    a[5] = 134.9634114+(477198.8676313+(0.008997+T/69699.0-T*T/14712000.0)*T)*T;
    /* mean obliqity of the ecliptic */
    a[6] = (84381.448-(46.8150+(0.00059-0.001813*T)*T)*T)/3600.0;
    /* mean eccentricity */
    a[7] = 0.016708617 - (0.000042037 + 0.0000001236*T)*T;
 
    for (i=0; i<7; i++) {
        a[i] = modf(a[i]/360.0, &ip);
        if (a[i] < 0.0) a[i] += 1.0;
	a[i] *= (2.0*PI);
    }
}
 
/*
 * Calculation of nutation matrix
 * input:
 *     Js:                 Julian date of epoch
 * output:
 *     NutMat:             matrix describing nutation at Js
 */
void nut(double Js, rotMatrix NutMat)
{
    double T, eps, l, lp, w, D, F, ip, arg;
    double deps, dPsi;
 
    T = (Js-2451545.0)/36525.0;
    /* mean obliquity of the ecliptic */
    eps = 84381.448-46.8150*T-0.00059*T*T+0.001813*T*T*T;
    eps = eps/SECPERREV*(2.0*PI);
    /* mean anomaly of the Moon */
    l = 485866.733+(1325.0*1296000.0+715922.633)*T+31.310*T*T+0.064*T*T*T;
    l = modf(l/SECPERREV,&ip)*(2.0*PI);
    /* mean anomaly of the Sun (Earth) */
    lp = 1287099.804+(99.0*1296000.0+1292581.244)*T-0.577*T*T-0.012*T*T*T;
    lp = modf(lp/SECPERREV,&ip)*(2.0*PI);
    /* difference L-w, where L mean longitude of the Moon */
    F = 335778.877+(1342.0*1296000.0+295263.137)*T-13.257*T*T+0.011*T*T*T;
    F = modf(F/SECPERREV,&ip)*(2.0*PI);
    /* mean elongation of the Moon from the Sun */
    D = 1072261.307+(1236.0*1296000.0+1105601.328)*T-6.891*T*T+0.019*T*T*T;
    D = modf(D/SECPERREV,&ip)*(2.0*PI);
    /* longitude of the ascending node of the Moon */
    w = 450160.280-(5.0*1296000.0+482890.539)*T+7.455*T*T+0.008*T*T*T;
    w = modf(w/SECPERREV,&ip)*(2.0*PI);
 
    arg = w;              dPsi = -17.1996*sin(arg);  deps =   9.2025*cos(arg);
    arg = 2.0*w;          dPsi +=  0.2062*sin(arg);  deps += -0.0895*cos(arg);
    arg = 2.0*(F-D+w);    dPsi += -1.3187*sin(arg);  deps +=  0.5736*cos(arg);
    arg = lp;             dPsi +=  0.1426*sin(arg);  deps +=  0.0054*cos(arg);
    arg = lp+2.0*(F-D+w); dPsi += -0.0517*sin(arg);  deps +=  0.0224*cos(arg);
    arg = 2.0*(F-D+w)-lp; dPsi +=  0.0217*sin(arg);  deps += -0.0095*cos(arg);
    arg = 2.0*(F-D)+w;    dPsi +=  0.0129*sin(arg);  deps += -0.0070*cos(arg);
    arg = 2.0*(F+w);      dPsi += -0.2274*sin(arg);  deps +=  0.0977*cos(arg);
    arg = l;              dPsi +=  0.0712*sin(arg);  deps += -0.0007*cos(arg);
    arg = 2.0*F+w;        dPsi += -0.0386*sin(arg);  deps +=  0.0200*cos(arg);
    arg = l+2.0*(F+w);    dPsi += -0.0301*sin(arg);  deps +=  0.0129*cos(arg);
    arg = l-2.0*D;        dPsi += -0.0158*sin(arg);  deps += -0.0001*cos(arg);
    arg = 2.0*(F+w)-l;    dPsi +=  0.0123*sin(arg);  deps += -0.0053*cos(arg);
 
    dPsi *= (2.0*PI)/SECPERREV;
    deps *= (2.0*PI)/SECPERREV;
 
    NutMat[0][0] =  1.0;
    NutMat[0][1] = -dPsi*cos(eps);
    NutMat[0][2] = -dPsi*sin(eps);
    NutMat[1][0] =  dPsi*cos(eps);
    NutMat[1][1] =  1.0;
    NutMat[1][2] = -deps;
    NutMat[2][0] =  dPsi*sin(eps);
    NutMat[2][1] =  deps;
    NutMat[2][2] =  1.0;
}
 
/*
 * Calculation of precession matrix
 * input:
 *     Js:                 Julian date of starting epoch
 *     Je:                 Julian date of ending epoch
 * output:
 *     PreMat:             matrix describing precession from Js to Je
 */
void pre(double Js, double Je, rotMatrix PreMat)
{
    double t,T;
    double zeta,z,theta;
 
    T = (Js-2451545.0)/36525.0;
    t = (Je-Js)/36525.0;
    zeta  = ( (2306.2181+1.39656*T-0.000139*T*T)*t
	     + (0.30188-0.000344*T)*t*t + 0.017998*t*t*t )*(2.0*PI)/SECPERREV;
    z     = ( (2306.2181+1.39656*T-0.000139*T*T)*t
	     + (1.09468+0.000066*T)*t*t + 0.018203*t*t*t )*(2.0*PI)/SECPERREV;
    theta = ( (2004.3109-0.85330*T-0.000217*T*T)*t
	     - (0.42665+0.000217*T)*t*t - 0.041833*t*t*t )*(2.0*PI)/SECPERREV;
    PreMat[0][0] =  cos(zeta)*cos(theta)*cos(z)-sin(zeta)*sin(z);
    PreMat[0][1] = -sin(zeta)*cos(theta)*cos(z)-cos(zeta)*sin(z);
    PreMat[0][2] = -sin(theta)*cos(z);
    PreMat[1][0] =  cos(zeta)*cos(theta)*sin(z)+sin(zeta)*cos(z);
    PreMat[1][1] = -sin(zeta)*cos(theta)*sin(z)+cos(zeta)*cos(z);
    PreMat[1][2] = -sin(theta)*sin(z);
    PreMat[2][0] =  cos(zeta)*sin(theta);
    PreMat[2][1] = -sin(zeta)*sin(theta);
    PreMat[2][2] =  cos(theta);
}
 
/*
 * Calculation of Greenwich mean sidereal time at 0h UT for given Julian date
 * input:
 *     JD:                 Julian Date
 * returns:
 *     GMST:               Greenwich mean sidereal time (revs)
 */
double GMST(double JD)
{
    double Tu, GHA, ip;
 
    Tu = (JD-2451545.0)/36525.0;
    GHA = 24110.54841+(8640184.812866+(0.093104-6.2e-6*Tu)*Tu)*Tu;
    GHA /= SECPERDAY;
    GHA = modf(GHA,&ip);
    if (GHA < 0.0) GHA += 1.0;
    return (GHA);
}
 
/*
 * Calculation of mean local sidereal time
 * input:
 *     Year, Month, Day:   date (yyyy,mm,dd)
 *     ZTime:              zone time (revolutions)
 *     ObsLong:            longitude (revolutions, east is positive) 
 *     ZtLong:             longitude of time zone (revolutions)
 * output:
 *     JulDate:            Julian Date for given time zone
 * returns:
 *                         mean sidereal time for given time zone (revs)
 */
double LMST(int Year, int Month, int Day, 
	    double ZTime, double ObsLong, double ZtLong, 
	    double *JulDate)
{
    double GHA, SidTime, ip;
 
    *JulDate = djl(Year,Month,Day);
    GHA = GMST(*JulDate);
    SidTime = modf(GHA+ObsLong+1.0027379093*(ZTime+ZtLong),&ip);
    *JulDate += (ZTime+ZtLong);
    return (SidTime);
}
 
/* Julian epoch for given Julian date */
double JEpoch(double JD)
{
    return (2000.0+(JD-2451545.0)/365.25);
}
 
/* Besselian epoch for given Julian date */
double BEpoch(double JD)
{
    return (1900.0+(JD-2415020.31352)/365.242198781);
}
 
