#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <sys/types.h>
#include <dirent.h>
#include <time.h>
#include <ctype.h>
#include "drp.h"
#include "options.h"
#include "fits.h"

#define READ 0
  
void scanio(char *, int , SCAN *);

static char scanname[MAXNAMLEN+1];

char PROGRAM_NAME[] = "bintable";
char description[] = "store scans as binary table file";
char required[] = "";
struct _options options[] = {
{ "-help",			"Print out this message" },
{ "-file filename",             "specify output filename" },
{NULL, NULL }};

void ParseOpts(int *pargc, char ***pargv)
{
  char *opt, *optarg;

  opt = (*pargv)[0] + 1;

  if (!strcmp(opt, "help")) {
    Help();
    exit(0);
  }

  /* file name */
  if (!strcmp(opt, "file")) {
    GetOption(&optarg, pargc, pargv);
    strcpy(scanname, optarg);
    return;
  }
  Syntax(**pargv);
}

#define FALSE        0
#define TRUE         1

#define BLANK   (-32768)

typedef short int WORD;

struct fitskey card;
  
void addFITScard(struct fitskey *);
int writeFITSheader(FILE *);
int writeFITSdata(FILE *,int ,int ,void *);
extern int used;

void VoidCard(int index)
{
  card.key = index;
  card.dim = 0;
  addFITScard(&card);
}

void BoolCard(int index, int b)
{
  card.key = index;
  card.dim = 0;
  card.val.l = (long)b;
  addFITScard(&card);
}

void CharCard(int index, int dim, char *s)
{
  int i, len;

  card.key = index;
  card.dim = dim;
  strncpy(card.val.str, s, 16);
  len = strlen(card.val.str);
  if (len < 8) {
    for (i = len; i < 8; i++) card.val.str[i] = ' ';
    card.val.str[8] = '\0';
  }
  addFITScard(&card);
}

void LongCard(int index, int dim, long l)
{
  card.key = index;
  card.dim = dim;
  card.val.l = l;
  addFITScard(&card);
}

void RealCard(int index, int dim, double d)
{
  card.key = index;
  card.dim = dim;
  card.val.d = d;
  addFITScard(&card);
}

void KillJunk(char s[])
{
  int i,j;

  for (i=0, j=0; i<strlen(s); i++)
    if (isalnum(s[i])) s[j++] = s[i];
  s[j++] = '\0';
}

SCAN OnScan;

struct TableEntry {
  char ctype1[8];
  int maxis1;
  float crval1;
  float cdelt1;
  float crpix1;
  char ctype2[8];
  float crval2;
  float cdelt2;
  float crpix2;
  char ctype3[8];
  float crval3;
  float cdelt3;
  float crpix3;
  char object[8];
  char utc[8];
  char lst[8];
  char date[8];
  float scanno;
  float epoch;
  float tsys;
  float tau;
  float dbl;
  float obstime;
  float azimuth;
  float elevation;
  double restfreq;
  double vlsr;
  double deltav;
  float spectrum[MAXCHANNELS];
};

int main(int argc, char *argv[])
{
  FILE *fits;
  /*   char scannum[10], *path; */
  char string[20];
  /*   float max, min, Y; */
  /*   double bs, bz; */
  int i, j, n, col, size;
  struct TableEntry *data;
  struct tm *now;
  time_t clock;
  
  GetOpts(argc, argv);
  GetScan(&OnScan);

  if (scanname[0] == '\0') {
    strncpy(scanname, OnScan.Name, 12);
    KillJunk(scanname);
    strcat(scanname, ".btx");
  } else {
    argc -= 2;
    argv += 2;
  }
  argc--; /* don't count program name */
  argv++;
  if (!argc) DRPerror("no scans to store");

  fits = fopen(scanname, "w");
  if (fits == NULL) DRPerror("can't open file '%s'", scanname);
    
  time(&clock);
  now = localtime(&clock);
  sprintf(string, "%02d/%02d/%02d",
	  now->tm_mday, now->tm_mon+1, now->tm_year%100);

  used = 0;

  BoolCard(KW_SIMPLE, TRUE);
  LongCard(KW_BITPIX, 0, 8);
  LongCard(KW_NAXIS,  0, 0);
  BoolCard(KW_EXTEND, TRUE);
  CharCard(KW_ORIGIN, 0, "DRP bintable");
  CharCard(KW_DATE, 0, string);
  VoidCard(KW_END);
  writeFITSheader(fits);

  used = 0;

  CharCard(KW_XTENSION, 0, "BINTABLE");
  LongCard(KW_BITPIX,  0, 8);
  LongCard(KW_NAXIS, 0,  2);
  LongCard(KW_NAXIS, 1,  sizeof(struct TableEntry));
  LongCard(KW_NAXIS, 2,  argc);
  LongCard(KW_PCOUNT, 0, 0);
  LongCard(KW_GCOUNT, 0, 1);
  LongCard(KW_TFIELDS, 0, 29);
  CharCard(KW_EXTNAME, 0, "OSO SINGLE DISH");
  LongCard(KW_EXTVER, 0, 0);
  LongCard(KW_NMATRIX, 0, 1);
  CharCard(KW_BUNIT, 0, "K");
  LongCard(KW_MAXIS, 0, 3);

  size = 0;
  col = 1;

  CharCard(KW_TFORM, col, "8A      ");   size += 8*sizeof(char);
  CharCard(KW_TTYPE, col, "CTYPE1  ");
  col++;

  CharCard(KW_TFORM, col, "1I      ");   size += sizeof(int);
  CharCard(KW_TTYPE, col, "MAXIS1  ");
  col++;

  CharCard(KW_TFORM, col, "1E      ");   size += sizeof(float);
  CharCard(KW_TTYPE, col, "CRVAL1  ");
  col++;

  CharCard(KW_TFORM, col, "1E      ");   size += sizeof(float);
  CharCard(KW_TTYPE, col, "CDELT1  ");
  col++;

  CharCard(KW_TFORM, col, "1E      ");   size += sizeof(float);
  CharCard(KW_TTYPE, col, "CRPIX1  ");
  col++;

  LongCard(KW_MAXIS, 2, 1);
  CharCard(KW_TFORM, col, "8A      ");   size += 8*sizeof(char);
  CharCard(KW_TTYPE, col, "CTYPE2  ");
  col++;

  CharCard(KW_TFORM, col, "1E      ");   size += sizeof(float);
  CharCard(KW_TTYPE, col, "CRVAL2  ");
  col++;

  CharCard(KW_TFORM, col, "1E      ");   size += sizeof(float);
  CharCard(KW_TTYPE, col, "CDELT2  ");
  col++;

  CharCard(KW_TFORM, col, "1E      ");   size += sizeof(float);
  CharCard(KW_TTYPE, col, "CRPIX2  ");
  col++;

  LongCard(KW_MAXIS, 3, 1);
  CharCard(KW_TFORM, col, "8A      ");   size += 8*sizeof(char);
  CharCard(KW_TTYPE, col, "CTYPE3  ");
  col++;

  CharCard(KW_TFORM, col, "1E      ");   size += sizeof(float);
  CharCard(KW_TTYPE, col, "CRVAL3  ");
  col++;

  CharCard(KW_TFORM, col, "1E      ");   size += sizeof(float);
  CharCard(KW_TTYPE, col, "CDELT3  ");
  col++;

  CharCard(KW_TFORM, col, "1E      ");   size += sizeof(float);
  CharCard(KW_TTYPE, col, "CRPIX3  ");
  col++;

  CharCard(KW_TFORM, col, "8A      ");   size += 8*sizeof(char);
  CharCard(KW_TTYPE, col, "OBJECT  ");
  col++;

  CharCard(KW_TFORM, col, "8A      ");   size += 8*sizeof(char);
  CharCard(KW_TTYPE, col, "UTC     ");
  col++;

  CharCard(KW_TFORM, col, "8A      ");   size += 8*sizeof(char);
  CharCard(KW_TTYPE, col, "LST     ");
  col++;

  CharCard(KW_TFORM, col, "8A      ");   size += 8*sizeof(char);
  CharCard(KW_TTYPE, col, "DATE    ");
  col++;

  CharCard(KW_TFORM, col, "1E      ");   size += sizeof(float);
  CharCard(KW_TTYPE, col, "SCAN-NUM");
  col++;

  CharCard(KW_TFORM, col, "1E      ");   size += sizeof(float);
  CharCard(KW_TTYPE, col, "EPOCH   ");
  col++;

  CharCard(KW_TFORM, col, "1E      ");   size += sizeof(float);
  CharCard(KW_TTYPE, col, "TSYS    ");
  CharCard(KW_TUNIT, col, "K       ");
  col++;

  CharCard(KW_TFORM, col, "1E      ");   size += sizeof(float);
  CharCard(KW_TTYPE, col, "TAU-ATM ");
  col++;

  CharCard(KW_TFORM, col, "1E      ");   size += sizeof(float);
  CharCard(KW_TTYPE, col, "DBLOAD  ");
  col++;

  CharCard(KW_TFORM, col, "1E      ");   size += sizeof(float);
  CharCard(KW_TTYPE, col, "OBSTIME ");
  col++;

  CharCard(KW_TFORM, col, "1E      ");   size += sizeof(float);
  CharCard(KW_TTYPE, col, "AZIMUTH ");
  col++;

  CharCard(KW_TFORM, col, "1E      ");   size += sizeof(float);
  CharCard(KW_TTYPE, col, "ELEVATIO");
  col++;

  CharCard(KW_TFORM, col, "1D      ");   size += sizeof(double);
  CharCard(KW_TTYPE, col, "RESTFREQ");
  col++;

  CharCard(KW_TFORM, col, "1D      ");   size += sizeof(double);
  CharCard(KW_TTYPE, col, "VLSR    ");
  col++;

  CharCard(KW_TFORM, col, "1D      ");   size += sizeof(double);
  CharCard(KW_TTYPE, col, "DELTAV  ");
  col++;

  CharCard(KW_TFORM, col, "2048E   ");   size += 2048*sizeof(float);
  CharCard(KW_TTYPE, col, "SPECTRUM");
  CharCard(KW_TDIM,  col, "(2048,1,1)");
  col++;

  VoidCard(KW_END);

  writeFITSheader(fits);

  n = argc;
  printf("allocate %d binary table entries of size %d\n", n, size);
  printf("size of struct TableEntry = %ld\n", sizeof(struct TableEntry));
  data = (struct TableEntry *)calloc(n, sizeof(struct TableEntry));
  if (data == NULL) DRPerror("memory allocation failure");

  for (i = 0; i < argc; i++) {
    printf("next scan '%s'\n", *argv);
    scanio(*argv, READ, &OnScan);
    strncpy(data[i].ctype1, "FREQ    ", 8);
    data[i].maxis1 = OnScan.NChannel;
    data[i].crval1 = 0.0;
    data[i].cdelt1 = (float)OnScan.FreqRes;
    data[i].crpix1 = (float)(CenterCh(&OnScan)+1);
    strncpy(data[i].ctype2, "RA---SIN", 8);
    data[i].crval2 = OnScan.Longitude;
    data[i].cdelt2 = 0.0;
    data[i].crpix2 = 1.0;
    strncpy(data[i].ctype3, "DEC--SIN", 8);
    data[i].crval3 = OnScan.Latitude;
    data[i].cdelt3 = 0.0;
    data[i].crpix3 = 1.0;
    strncpy(data[i].object, OnScan.Name, 8);
    sprintf(data[i].utc, "%02d:%02d:%02d",
	    OnScan.UTHour, OnScan.UTMin, OnScan.UTSec);
    sprintf(data[i].lst, "%02d:%02d:%02d",
	    OnScan.STHour, OnScan.STMin, OnScan.STSec);
    sprintf(data[i].date, "%02d/%02d/%02d",
	    OnScan.Day, OnScan.Month, OnScan.Year%100);
    data[i].scanno = OnScan.ScanNo;
    data[i].epoch = OnScan.Equinox;
    data[i].tsys = OnScan.Tsys;
    data[i].tau = OnScan.Tau;
    data[i].dbl = OnScan.dBl;
    data[i].obstime = OnScan.IntTime;
    data[i].azimuth = OnScan.Azimuth*180.0/PI;
    data[i].elevation = OnScan.Elevation*180.0/PI;
    data[i].restfreq = OnScan.RestFreq*1.0e6;
    data[i].vlsr = OnScan.VSource*1.0e3;
    data[i].deltav = OnScan.VelRes*1.0e3;
    for (j = 0; j < OnScan.NChannel; j++)
      data[i].spectrum[j] = OnScan.Channels[j];
    argc--;
    argv++;
  }
    
  i = writeFITSdata(fits, n, size, (void *)data);
  DRPinfo("scan written to file '%s'", scanname);

  exit(0);
}
