#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "drp.h"

#define FITSKNOWN
#include "cfits.h"

#define TRUE  1
#define FALSE 0

static char record[PHYSRECLEN];
static char keyword[KEYWORDLEN+1];
static char line[LOGRECLEN+1];
static char variable[MAXVARLEN+1];
static char *value;
static char *current;
static int allocated;

struct fitskey *kw;
int used;

#define CHUNKSIZE (PHYSRECLEN/LOGRECLEN)

void AllocKW(void)
{
    if (kw == NULL) {
	allocated = CHUNKSIZE;
	kw = (struct fitskey *)malloc(allocated*sizeof(struct fitskey));
    } else {
	allocated += CHUNKSIZE;
	kw = (struct fitskey *)realloc(kw, allocated*sizeof(struct fitskey));
    }
    if (kw == (struct fitskey *)NULL) {
	DRPerror("memory allocation failure");
    }
#ifdef DEBUG
    printf("used = %d, allocated = %d\n", used, allocated);
#endif
}

int extractFITScard(void)
{
    char *ptr, *quote;
    int i, len, nkw;

    ptr = current;
    strncpy(line, current , LOGRECLEN);    /* get next 80 characters        */
    line[LOGRECLEN] = '\0';                /* place an end of string        */
    strncpy(keyword, line, KEYWORDLEN);    /* get first 8 characters        */
    keyword[KEYWORDLEN] = '\0';            /* terminate keyword             */

    ptr = current+KEYWORDLEN;              /* point past keyword            */
    if (strncmp(ptr, "= ", 2) == 0) {      /* is if followed by '= '?       */
	ptr += 2;                          /* point past '= '               */
	value = ptr;
	i = 0;                             /* count chars. in value field   */
        /* in the following while loop we collect all characters from the   */
	/* value field following the '= '. We allow MAXVARLEN characters at */
        /* most. The loop takes care of enclosing quotes, in case the value */
        /* is a string.                                                     */
	while (i < MAXVARLEN) {            /* extract value field           */
	    switch (*ptr) {
	      case '\'':                   /* quote encountered -> string   */
		ptr++;                     /* point past quote              */
		do {
		    quote = strchr(ptr, '\''); /* we expect a second quote  */
		    if (quote == NULL) return (-1);  /* else error ...      */
		    *quote++ = '\0';       /* put end of string character   */
		    strcpy(&variable[i], ptr);  /* copy string              */
		    i += strlen(ptr);      /* count characters in string    */
		    /* check if string contains a quote */
		    if (*quote == '\'') {
			variable[i++] = '\'';
			ptr = quote+1;
		    }
		} while ((*quote == '\'') && i < MAXVARLEN);
#ifdef DEBUG
		if (strlen(variable) < KEYWORDLEN)
		  DRPwarning("non-standard string variable");
#endif
		i = MAXVARLEN;
		break;
	      case '/':                    /* start of comment              */
		variable[i] = '\0';        /* put end of string             */
		i = MAXVARLEN;             /* set condition for while loop  */
		break;
	      default:
		variable[i++] = *ptr++;    /* count character               */
		if (i == MAXVARLEN) variable[i] = '\0'; /* too many ?       */
		break;
	    }
	}
    } else {
	value = NULL;
    }
#ifdef DEBUG
    printf("keyword = %.8s, value = %s\n", keyword, variable);
#endif

    if (used == allocated) AllocKW();

    /* Now, 'keyword' holds the FITS keyword string and 'variable' possibly */
    /* holds the value of the associated variable as an alphabetic or nume- */
    /* ric string. Loop through all known keywords and compare with what we */
    /* have read. Once we find a match, we know how to interpret the value  */
    /* field.                                                               */
    nkw = sizeof(known)/sizeof(struct knownkey);
    for (i = 0; i < nkw; i++) {
	len = strlen(known[i].text);
	if (strncmp(keyword, known[i].text, len) == 0) { /* known ?    */
	    /*
	      if (i == KW_COMMENT) printf("%s\n", line);
	      if (i == KW_HISTORY) printf("%s\n", line); 
	      */
	    kw[used].key = i;
	    ptr = &keyword[len];
	    if (isdigit(*ptr)) kw[used].dim = atoi(ptr);
	    else               kw[used].dim = atoi(ptr);

#ifdef DEBUG
	    printf("keyword: %d -- dim: %d\n", kw[used].key, kw[used].dim);
#endif
	    /* if this keyword requires a value, but we didn't see any, we
               are in trouble                                               */
	    if ((known[i].typ != NVARTYPE) && (value == NULL)) return (-1);

	    /* take action depending on variable type                       */
	    switch (known[i].typ) {
	      case NVARTYPE:             /* no value required               */
		break;
	      case BOOLTYPE:             /* logical variable                */
		if (variable[19] == 'T') kw[used].val.l = TRUE;
		else                     kw[used].val.l = FALSE;
		break;
	      case CHARTYPE:             /* character string                */
		strncpy((char *)kw[used].val.str, variable, KEYWORDLEN*2);
		break;
	      case LONGTYPE:             /* long integer                    */
		kw[used].val.l = atoi(variable);
		break;
	      case REALTYPE:             /* float                           */
		kw[used].val.d = atof(variable);
		break;
	      case CPLXTYPE:             /* complex                         */
		kw[used].val.c[0] = atof(variable);
		kw[used].val.c[1] = atof(variable+20);
		break;
	    }
	    used++;
	    return (i);
	}
    }
    /* here we get for unknown keywords                                     */
#ifdef DEBUG
    DRPwarning("unknown keyword '%s'", keyword);
#endif
    return (0);
}
    
struct fitskey *readFITSheader(FILE *fits)
{   
    int len, key;

    /* read one record of 2880 bytes                                        */
    len = fread(record, sizeof(char), PHYSRECLEN, fits);
    if (len != PHYSRECLEN) {                  /* test for failure           */
	DRPwarning("error reading header record");
	return (NULL);
    }
    current = record;                         /* point to start of record   */
    
    do {
	/* If all characters are used up, read some more.                   */
	if ((current - record) >= PHYSRECLEN) {
	    len = fread(record, sizeof(char), PHYSRECLEN, fits);
	    if (len != PHYSRECLEN) {
		DRPwarning("error reading header record");
		return (NULL);
	    }
	    current = record;
	}
	/* Interpret what we have read so far by calling 'extractFITScard'. */
	if ((key = extractFITScard()) == -1) {
	    DRPwarning("error parsing card");
	    return (NULL);
	}
	/* we have parsed one logical record, update pointer accordingly    */
	current += LOGRECLEN;
    } while (key != KW_END);     /* until we have seen 'END'   */
    
    return (kw);
}

int readFITSdata(FILE *fits, int nobjs, int size, void *data)
{
    int bytes, i, len, left;
    char *ptr;

    ptr = (char *)data;
    bytes = size*nobjs;
    i = 0;
    while (i < bytes) {
	if ((i % PHYSRECLEN) == 0) {
#ifdef DEBUG
	    printf("reading data record\n");
#endif
	    len = fread(record, sizeof(char), PHYSRECLEN, fits);
	    if (len != PHYSRECLEN) {
		DRPwarning("error reading data record");
		return (0);
	    }
	}
	left = bytes - i;
	if (left > PHYSRECLEN) left = PHYSRECLEN;
	memcpy(ptr, record, left);
	ptr += left;
	i += left;
    }
    return (bytes);
}

static int card;

void addFITScard(struct fitskey *fk)
{
    if (used == allocated) AllocKW();
    memcpy(&kw[used], fk, sizeof(struct fitskey));
#ifdef DEBUG
    printf("card %d '%s'\n", used, known[kw[used].key].text);
#endif
    used++;
}

void insertFITScard(struct fitskey *fk)
{
    int i, n;

    n = 0;
    if (fk->dim == 0) sprintf(keyword, "%s", known[fk->key].text);
    else   sprintf(keyword, "%s%d", known[fk->key].text, fk->dim);

    strcpy(line, keyword);
    n = strlen(keyword);
    for (i = n; i < KEYWORDLEN; i++) line[n++] = ' ';

    if (known[fk->key].typ != NVARTYPE) {
	line[n++] = '=';
	line[n++] = ' ';
	switch (known[fk->key].typ) {
	  case BOOLTYPE:
	    while (n < 29) line[n++] = ' ';
	    if (fk->val.l) line[n++] = 'T';
	    else                line[n++] = 'F';
	    break;
	  case CHARTYPE:
	    line[n++] = '\'';
	    for (i = 0; i < strlen(fk->val.str); i++) {
		line[n++] = fk->val.str[i];
	    }
	    line[n++] = '\'';
	    break;
	  case LONGTYPE:
	    sprintf(variable, "%20ld", fk->val.l);
	    for (i = 0; i < 20; i++) {
		line[n++] = variable[i];
	    }
	    break;
	  case REALTYPE:
	    sprintf(variable, "%20.12le", fk->val.d);
	    for (i = 0; i < 20; i++) {
		line[n++] = variable[i];
	    }
	    break;
	  case CPLXTYPE:
	    sprintf(variable, "%20.12le%20.12le", fk->val.c[0], fk->val.c[1]);
	    for (i = 0; i < MAXVARLEN; i++) {
		line[n++] = variable[i];
	    }
	    break;
	}
    }

    while (n < LOGRECLEN) line[n++] = ' ';

/*     line[LOGRECLEN-1] = '\n'; */


    line[LOGRECLEN] = '\0';

    strncpy(&record[card*LOGRECLEN], line, LOGRECLEN);
    card++;
#ifdef DEBUG
    printf("%s", line);
#endif    
}

int writeFITSheader(FILE *fits)
{   
    int len, i;

    card = 0;
    for (i = 0; i < used; i++) {
	insertFITScard(&kw[i]); 
	if (card == (PHYSRECLEN/LOGRECLEN)) {
#ifdef DEBUG
	    printf("writing header record\n");
#endif
	    len = fwrite(record, sizeof(char), PHYSRECLEN, fits);
	    if (len != PHYSRECLEN) {
		DRPwarning("error writing header record");
		return (-1);
	    }
	    card = 0;
	}
    }
    memset(line, ' ', LOGRECLEN-1);
    line[LOGRECLEN-1] = '\n';
    line[LOGRECLEN] = '\0';
    if (card) {
	while (card < (PHYSRECLEN/LOGRECLEN)) {
	    strncpy(&record[card*LOGRECLEN], line, LOGRECLEN);
	    card++;
#ifdef DEBUG
	    printf("%s", line);
#endif    
	}
#ifdef DEBUG
	printf("writing header record\n");
#endif
	len = fwrite(record, sizeof(char), PHYSRECLEN, fits);
	if (len != PHYSRECLEN) {
	    DRPwarning("error writing header record");
	    return (-1);
	}
	card = 0;
    }
    return (0);
}

int writeFITSdata(FILE *fits, int nobjs, int size, void *data)
{
    int bytes, i, len, left;
    char *ptr;

    ptr = (char *)data;
    bytes = size*nobjs;
    i = 0;
    while (i < bytes) {
	memset(record, '\0', PHYSRECLEN);
	left = bytes - i;
	if (left > PHYSRECLEN) left = PHYSRECLEN;
	memcpy(record, ptr, left);
	ptr += left;
	i += left;
#ifdef DEBUG
	printf("writing data record (%d of %d)\n", i, bytes);
#endif
	len = fwrite(record, sizeof(char), PHYSRECLEN, fits);
	if (len != PHYSRECLEN) {
	    DRPwarning("error writing data record (%d)", len);
	    return (0);
	}
    }
    return (bytes);
}

#ifdef BYTESWAP
/* the following routine is needed for Intel type CPUs in order
to have the proper byte gender required by the FITS standard */

void swapbytes(char *p, int n)
{
    char swap;
    int i;

    for (i = 0; i < n/2; i++) {
	swap = p[i];
	p[i] = p[n-1-i];
	p[n-1-i] = swap;
    }
}
#endif
