#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "plplot.h"
#include "drp.h"

#define TRUE 1
#define FALSE 0

extern int CtrLabs;
extern float *height, BLANK;
extern SCAN OnScan;
extern int smooth;
 
#define HEAD 0
#define TAIL 1

/* 
   The following structure serves to sort segments of a contour line.
   '*succ' is a pointer that holds the address of the succeeding segment
   after sorting. 'startx, starty' and 'stopx, stopy' are the start and
   stop coordinates of the segment, respectively. 
*/
struct segment {
   struct segment *succ;
   float startx, starty;
   float stopx, stopy;
};
  
/* Turn a segment around by interchanging start and stop coordinates */
void reverse (struct segment *l)
{
    float swap;
    swap = l->stopx; l->stopx = l->startx; l->startx = swap;
    swap = l->stopy; l->stopy = l->starty; l->starty = swap;
}
  
#define EPS 0.0001
 
/* 
   Check if two segments 'l1' and 'l2' join. 
   First check if start coordinates of 'l2' agree with start coordinates 
   of 'l1'. If yes return TRUE.
   If 'l1' and 'l2' have the same stop coordinates and parameter 'where'
   is set to TAIL, turn 'l2' around and return TRUE.
   If 'l1' and 'l2' have the same start coordinates and parameter 'where'
   is set to HEAD, turn 'l1' around and return TRUE.
   Checking the agreement of coordinates is only done with an accuracy
   of EPS.
*/
int join(struct segment *l1, struct segment *l2, int where)
{
    if (fabs(l1->stopx - l2->startx) < EPS)
      if (fabs(l1->stopy - l2->starty) < EPS) {
	  return (TRUE);
      }
    if (where == TAIL) {
	if (fabs(l1->stopx - l2->stopx) < EPS)
	  if (fabs(l1->stopy - l2->stopy) < EPS) {
	      reverse (l2);
	      return (TRUE);
	  }
    } else if (where == HEAD) {
	if (fabs(l1->startx - l2->startx) < EPS)
	  if (fabs(l1->starty - l2->starty) < EPS) {
	      reverse (l1);
	      return (TRUE);
	  }
    }
    return (FALSE);
}
  
/* 
   Remove segment 'l' from a linked list starting at 'head'.
*/
void remseg (struct segment *l, struct segment *head)
{
    struct segment *next;

    /* Starting at head, traverse through the linked list. */  
    for (next=head; next!=NULL; next=next->succ) {
	/* If a segment is pointing to the segment to be removed, 
	   make it point to the next one instead. */
	if (next->succ == l) {
	    next->succ = l->succ;
	    break;
	}
    }
}

/*
   This routine draws a straight line for each contour segment.
   The first segment is 'l'. There are 'n' segments to draw.
*/

void plsegs(struct segment *l, int n)
{
    int i;
    PLFLT *x, *y;
  
    x = (PLFLT *)calloc(n+1, sizeof(PLFLT));
    y = (PLFLT *)calloc(n+1, sizeof(PLFLT));
    if ((x == NULL) || (y == NULL)) DRPerror("memory allocation failure");

    x[0] = l->startx; 
    y[0] = l->starty;
    for (i = 0; i < n; i++) {
	x[i+1] = l->stopx; 
	y[i+1] = l->stopy;
	l = l->succ;
    }
    plline(n+1, x, y);
    free(y);
    free(x); 
}

/* 
   This routine may be used if you want smoothly curved lines for
   each segment instead of straight ones.
*/   

#define NSPLINE 10

void spline(struct segment *l, int n)
{
    struct segment *h;
    float xA, xB, xC, xD;
    float yA, yB, yC, yD;
    float a0, a1, a2, a3, b0, b1, b2, b3;
    float t;
    PLFLT *x = NULL, *y = NULL;
    int i, j, m;
    int closed;

    /* for just 1 line segment this is overkill */ 
    if (n == 1) {
	plsegs(l, n);
	return;
    }

    h = l;
    for (i = 0; i < n-1; i++) l = l->succ;
  
    xB = h->startx; yB = h->starty;
    xC = h->stopx;  yC = h->stopy;
    closed = join(l, h, HEAD);

    if (closed) {
	xA = l->startx; yA = l->starty;
	n += 1; 
    } else {
 	xA = 2*xB-xC;   yA = 2*yB-yC;
    }

    x = (PLFLT *)calloc(n*NSPLINE, sizeof(PLFLT));
    y = (PLFLT *)calloc(n*NSPLINE, sizeof(PLFLT));
    m = 0; l = h;
    for (i = 0; i < n; i++) {
	if (!closed && i == n-1) {
	    /* treat end point for unclosed contours */
	    xD = 2*xC-xB; yD = 2*yC-yB;
	} else {
	    /* let tail point back to head for closed contours */
	    if (closed && i == n-2) l = h;
	    else 	            l = l->succ;
	    xD = l->stopx; yD = l->stopy;  
	}
/*
	printf("x: (%f,%f,%f,%f)\n", xA, xB, xC, xD);
	printf("y: (%f,%f,%f,%f)\n", yA, yB, yC, yD);
*/
	a3 = (-xA+3*(xB-xC)+xD)/6.0; b3 = (-yA+3*(yB-yC)+yD)/6.0;
	a2 = (xA-2*xB+xC)/2.0; b2 = (yA-2*yB+yC)/2.0;
	a1 = (xC-xA)/2.0; b1 = (yC-yA)/2.0;
	a0 = (xA+4*xB+xC)/6.0; b0 = (yA+4*yB+yC)/6.0;
	for (j = 0; j < NSPLINE; j++) {
	    t = (float)j/(float)(NSPLINE+1);
	    x[m] = ((a3*t+a2)*t+a1)*t+a0;
	    y[m] = ((b3*t+b2)*t+b1)*t+b0;
	    m++;
	}
	xA = xB; xB = xC; xC = xD;
	yA = yB; yB = yC; yC = yD;
    }
    plline(m, x, y);
    free(y);
    free(x);
}

/* 
   Contour drawing routine with sorting of line segments by using
   linked lists. This avoids a lot of pen movements if the contours
   are to be drawn on a pen plotter. For a terminal the sorting isn't
   really necessary, but doesn't hurt.
*/
void contour(float level)
{
    struct segment *head, *next, *hd, *tl;
    int ix, iy, ic, n, index;
    int imin, imax, jmin, jmax, DimX;
    double stepx, stepy, tilt, st, ct;
    float corner[5], delta, x[2], y[2];

    /* 
      Initialize the linked list. We are using the space taken up by
      the channel values of our 'OnScan' spectrum, which are not needed.
      On other machines we could just allocate the space needed, but with
      the RTE-A we have to squeeze everything into 32k words.
      */
    head = NULL;
    next = (struct segment *)OnScan.Channels;

    /*  From the header of 'OnScan' get the mapping parameters. */
    stepx = OnScan.StepX; stepx *= RADTOMIN;
    stepy = OnScan.StepY; stepy *= RADTOMIN;
    tilt  = OnScan.PosAngle; ct = cos(tilt); st = sin(tilt);
  
    /* The dimensions of our map where stored in the 'flag' area by
       routine 'scanlog'. */
    imin = OnScan.flag[0];
    imax = OnScan.flag[1];
    jmin = OnScan.flag[2];
    jmax = OnScan.flag[3];
    DimX = imax-imin+1;
 
    /* Step through all the points of the grid. */
    for (iy=jmin; iy<jmax; iy++) {
	for (ix=imin; ix<imax; ix++) {
	    /* The grid is actually stored as a linear array, so calculate
	       the index for that array. */
	    index = (ix-imin)+(iy-jmin)*DimX;
	    /* Lookup the intensity at the current grid point (corner[0]),
	       the one to the right (corner[1]), the one to the right and up
	       (corner[2]) and the one up (corner[3]). For convenience further
	       down, set up a 5th corner equal to the 1st one. */
	    corner[0] = height[index];        if (corner[0] == BLANK) continue;
	    corner[1] = height[index+1];      if (corner[1] == BLANK) continue;
	    corner[2] = height[index+DimX+1]; if (corner[2] == BLANK) continue;
	    corner[3] = height[index+DimX];   if (corner[3] == BLANK) continue;
	    corner[4] = corner[0];
	 
	    /* 'n' counts the number of sides of the square formed by corner[0]
	       to corner[3], which are crossed by a contour at level 'level' */
	    n = 0;		/* initialize */
         
	    /* For each side between two corners from above, find out if 'level'
	       falls between the values at the corner, which means that a 
	       contour should cross that side. */
	    for (ic=0; ic<4; ic++) {
		if (fabs((double)(corner[ic+1]-corner[ic])) < EPS) delta = BLANK;
		else delta = (level-corner[ic])/(corner[ic+1]-corner[ic]);
		if (delta >= 0.0 && delta <= 1.0) {
		    /* Calculate the coordinates of the point on side 'ic' which
		       corresponds to 'level' by linearly interpolating between
		       the corresponding corners. */
		    switch (ic) {
		      case 0:	/* crossing between corner[0] and corner[1] */
			x[n] = (ix+delta)*stepx*ct+iy*stepy*st;
			y[n] = iy*stepy*ct-(ix+delta)*stepx*st;
			break;
		      case 1:	/* crossing between corner[1] and corner[2] */
			x[n] = (ix+1)*stepx*ct+(iy+delta)*stepy*st;
			y[n] = (iy+delta)*stepy*ct-(ix+1)*stepx*st;
			break;
		      case 2:	/* crossing between corner[2] and corner[3] */
			x[n] = (ix+1-delta)*stepx*ct+(iy+1)*stepy*st;
			y[n] = (iy+1)*stepy*ct-(ix+1-delta)*stepx*st;
			break;
		      case 3:	/* crossing between corner[3] and corner[4=0] */
			x[n] = ix*stepx*ct+(iy+1-delta)*stepy*st;
			y[n] = (iy+1-delta)*stepy*ct-ix*stepx*st;
			break;
		    }
		    n++;	/* a crossing has been found, count it */
		    /* If we already have two crossings, set up a segment
		       from one crossing to the other */                 
		    if (n == 2) {
			/* We could draw the segments now, if we don't care
			   about pen movements */
			/* drpmove(x[0],y[0]); drpdraw(x[1],y[1]); */
	       
			/* Instead we build a linked list of segments, which
			   will be sorted later. */
			/* Remember the old start of the list      */
			hd = head;   
			/* The new segment is stored at 'next', it will be the new
			   start of our list. */
			head = next; 
			/* 'next' is no longer needed and is incremented to point
			   point to more free space for future segments */
			next++;
			/* The new 'head' should point to the old 'head' */      
			head->succ = hd;
			/* Fill in the coordinates of start and stop points. */      
			head->startx = x[0];
			head->starty = y[0];
			head->stopx  = x[1];
			head->stopy  = y[1];
			/* Reset the count of crossings. */
			n = 0;
		    } 
		}
	    }
	}
    }

    /* Now we have all line segments for a given level in a linked list.
       We want to sort that list so that the contours may be drawn with as few
       pen up, pen down movements as possible. */
    /* Traverse through the linked list until the list is empty (head == NULL) 
       In the process of sorting any segments which join, are removed
       from the unsorted list and entered into a sorted list. If no more
       segments can be found that join with the sorted ones, the sorted segments
       are drawn. The process is repeated until the unsorted list is empty. */
    while (head != NULL) {
	/* 'hd' is the head(start) of our list of sorted segments. */
	hd = head;		/* initialize with first element from unsorted list */
	/* 'tl' is the tail(end) of our list of sorted segments */
	tl = head;		/* initialize with first element from unsorted list */
      
	/* Go through the unsorted list to find segments that join either
	   at the head or at the start of our sorted segment list. 'n'
	   is counting the number of segments already joined. */
	for (n=1, next=head->succ; next!=NULL; next=next->succ) {
	    /* Check if segment 'next' joins at the head. If yes, remove it from 
	       the unsorted list and make it the new head of the sorted list. */
	    if (join(next, hd, HEAD)) {
		remseg(next, head); /* remove from unsorted list */
		next->succ = hd; /* point to head of sorted list */
		hd = next;	/* 'next' becomes new head of sorted list */
		next = head;	/* restart loop through unsorted list */
		n++;		/* count the segment */
	    }

	    /* Check if segment 'next' joins at the tail. If yes, remove it from 
	       the unsorted list and make it the new tail of the sorted list. */
	    else if (join(tl, next, TAIL)) {
		remseg(next, head); /* remove from unsorted list */
		next->succ = head->succ; /* reset loop through unsorted list */
		tl->succ = next; /* let old tail point to new tail */
		tl = next;	/* 'next' becomes new tail */
		head = tl;	/* reset loop through unsorted list */
		n++;		/* count the segment */
	    }
	}
	/* Draw the segments in our sorted list, either by fitting smoothly 
	   curved splines or straight lines for each segment. */
	if (smooth) spline(hd, n);          /* draw splines        */
	else        plsegs(hd, n);          /* draw straight lines */
	head = head->succ;	/* restart processing the unsorted list. */
    }
}


