#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <ctype.h>
#include <unistd.h>
#include "drp.h"
#include "plplot.h"
#include "options.h"
#include "fits.h"

#undef DEBUG

#define CONTR 1
#define SHADE 4
#define SPECT 5
    
#define YES  1
#define NO   0

void plsfile(FILE *);

static char cubename[MAXNAMLEN+1];
  
/* defaults */
int which = CONTR;
int overwrite = NO;
int frame = YES;
int draft = YES;
int title = NO;
char markfile[MAXNAMLEN+1] = "";
char NewTitle[80];
char keyword[10];
int plane;
int geo[3] = { 1, 2, 3};
double xMin, xMax;
int xUnit = VELOCITY;
float level = 0.0;
float dl = 0.0;

#define BLANK (-32768)

typedef short int WORD;
typedef long int LONG;

char PROGRAM_NAME[] = "cube";
char description[] = "produce different views of data cube from log file";
char required[] = "";
struct _options options[] = {
{ "-help",		"print out this message" },
{ "-file filename",     "specify output filename (default: SOURCENAME.CUBE" },
{ "-plane n",		"draw contours for plane n" },
{ "-start level",       "specify lowest level for contour maps" },
{ "-inc increment",     "specify level increments" },
{ "-title text",        "specify new title for plot" },
{ "-view x y z",        "specify projection" },
{ "-vel v1 v2",		"select velocity range" },
{ "-map what",          "what: contr, shade, spect" },
{ "-over",		"write over existing plot" },
{ "-nodraft",		"use extended font for high quality labels" },
{NULL, NULL }};

void ParseOpts(int *pargc, char ***pargv)
{
    char *opt, *optarg;

    opt = (*pargv)[0] + 1;

    if (!strcmp(opt, "help")) {
	Help();
	exit(EX_SUCCESS);
    }
    if (!strcmp(opt, "file")) {
	GetOption(&optarg, pargc, pargv);
	strcpy(cubename, optarg);
	return;
    }
    if (!strcmp(opt, "plane")) {
	GetOption(&optarg, pargc, pargv);
	plane = atoi(optarg);
	return;
    }
    if (!strcmp(opt, "start")) {
	GetOption(&optarg, pargc, pargv);
	level = atof(optarg);
	return;
    }
    if (!strcmp(opt, "inc")) {
	GetOption(&optarg, pargc, pargv);
	dl = atof(optarg);
	return;
    }
    if (!strcmp(opt, "title")) {
	title = YES;
	GetOption(&optarg, pargc, pargv);
	strcpy(NewTitle, optarg);
	return;
    }
    if (!strcmp(opt, "view")) {
 	GetOption(&optarg, pargc, pargv);
	geo[0] = atoi(optarg);
 	GetOption(&optarg, pargc, pargv);
	geo[1] = atoi(optarg);
 	GetOption(&optarg, pargc, pargv);
	geo[2] = atoi(optarg);
	return;
    }
    if (!strcmp(opt, "vel")) {
	GetOption(&optarg, pargc, pargv);
	xMin = atof(optarg);
	GetOption(&optarg, pargc, pargv);
	xMax = atof(optarg);
	xUnit = VELOCITY;
	return;
    }
    if (!strcmp(opt, "map")) {
	GetOption(&optarg, pargc, pargv);
	for (opt = optarg; *opt != '\0'; opt++) *opt = toupper(*opt);
	if (strncmp(optarg, "CONTR", 5) == 0) which = CONTR;
	if (strncmp(optarg, "SHADE", 5) == 0) which = SHADE;
	if (strncmp(optarg, "SPECT", 5) == 0) which = SPECT;
	return;
    }
    if (!strcmp(opt, "over")) {
	overwrite = YES;
	frame = NO;
	return;
    }
    if (!strcmp(opt, "nodraft")) {
	draft = NO;
	return;
    }
    Syntax(**pargv);
}

void KillJunk(char s[])
{
    int i,j;

    for (i=0, j=0; i<strlen(s); i++)
      if (isalnum(s[i])) s[j++] = s[i];
    s[j++] = '\0';
}
  
SCAN OnScan;
#define MAXDIMENSION 3
#define RATYPE   1
#define DECTYPE  2
#define VELTYPE  3
#define FREQTYPE 4
struct Axis {
    int dim;
    int type;
    char label[16];
    double pixel;
    double value;
    double delta;
    double rotate;
} axis[MAXDIMENSION];

extern int used;
struct fitskey *readFITSheader(FILE *);
int readFITSdata(FILE *, int ,int ,void *);
void showcube(float *);
#ifdef BYTESWAP
void swapbytes(char *, int );
#endif

void subcube(float *rp, int v_axis, int from, int to)
{
    int i, j, k;
    int new, old, size;
    static int range[3][2];
    
    size = 1;
    for (i = 0; i < 3; i++) {
	size *= axis[i].dim;
    }
#ifdef DEBUG
    printf("axis %d from %d to %d\n", v_axis, from, to);
#endif
    for (i = 0; i < 3; i++) {
	range[i][0] = 0;
	range[i][1] = axis[i].dim-1;
    }
    range[v_axis][0] = from;
    range[v_axis][1] = to;

    for (i = range[0][0]; i <= range[0][1]; i++) {
	for (j = range[1][0]; j <= range[1][1]; j++) {
	    for (k = range[2][0]; k <= range[2][1]; k++) {
		old = (k*axis[0].dim + j)*axis[1].dim + i;
		new = ((k-range[2][0])*axis[0].dim 
		       + (j-range[1][0]))*axis[1].dim 
			 + (i-range[0][0]);
		if (old < 0 || old > size) DRPerror("index out of range");
		if (new < 0 || new > size) DRPerror("index out of range");
		rp[new] = rp[old];
	    }
	}
    }
    for (i = 0; i < 3; i++) {
	axis[i].dim = range[i][1]-range[i][0]+1;
#ifdef DEBUG
	printf("new dimension[%d] = %d\n", i, axis[i].dim);
#endif
    }
}

int main(int argc, char *argv[])
{
    FILE *fits, *meta;
    static char metaname[MAXNAMLEN+1];
    void *data;
    WORD *ip;
    LONG *jp;
    float *rp;
    float bzero = 0.0, bscale = 0.0, blank = 0.0;
    int i, size;
    int /* simple, extend, */ bitpix = 0, naxis = 0;
    int v_axis = 0, from, to;
    char *path;
    struct fitskey *kw;
  
    GetOpts(argc, argv);

    if (cubename[0] == '\0') {
	if (argv[1] == NULL) {
	    Syntax("");
	    exit(EX_ARGSBAD);
	} else {
	    if ((path = getenv("FITSAREA")) == NULL) { 
		strcpy(cubename, argv[1]);
	    } else {
		strcpy(cubename, path);
		if (path[strlen(path)-1] != '/') strcat(cubename, "/");
		strcat(cubename, argv[1]);
	    }
	}
    }

    fits = fopen(cubename, "r");        /* open the file for reading  */
    if (fits == NULL) DRPerror("can't open file '%s'", cubename);

    used = 0;
    kw = readFITSheader(fits);
    if (kw == NULL) DRPerror("can't read fits header");

    for (i = 0; i < used; i++) {
#ifdef DEBUG
	printf("known keyword: %s [%d]\n", known[kw[i].key].text, kw[i].dim);
#endif
	switch (kw[i].key) {
	  case KW_SIMPLE:
	    /* simple = kw[i].val.l; */
	    break;
	  case KW_EXTEND:
	    /* extend = kw[i].val.l; */
	    break;
	  case KW_BITPIX:
	    bitpix = kw[i].val.l;
	    break;
	  case KW_NAXIS:
	    if (kw[i].dim == 0) naxis = kw[i].val.l;
	    if (naxis > MAXDIMENSION) DRPwarning("dimension too high");
	    else axis[kw[i].dim-1].dim = kw[i].val.l;
	    break;
	  case KW_CTYPE:
	    strncpy(axis[kw[i].dim-1].label, kw[i].val.str, 16);
	    if (strncmp(kw[i].val.str, "RA---SIN", 8) == 0) {
		axis[kw[i].dim-1].type = RATYPE;
	    }
	    if (strncmp(kw[i].val.str, "DEC--SIN", 8) == 0) {
		axis[kw[i].dim-1].type = DECTYPE;
	    }
	    if (strncmp(kw[i].val.str, "VELO-LSR", 8) == 0) {
		axis[kw[i].dim-1].type = VELTYPE;
		v_axis = kw[i].dim-1;
	    }
	    if (strncmp(kw[i].val.str, "FREQ", 4) == 0) {
		axis[kw[i].dim-1].type = FREQTYPE;
	    }
	    break;
	  case KW_CRPIX:
	    axis[kw[i].dim-1].pixel = kw[i].val.d;
	    break;
	  case KW_CRVAL:
	    axis[kw[i].dim-1].value = kw[i].val.d;
	    if (axis[kw[i].dim-1].type == VELTYPE)
	      axis[kw[i].dim-1].value /= 1000.0; /* convert m/s to km/s */
	    break;
	  case KW_CDELT:
	    axis[kw[i].dim-1].delta = kw[i].val.d;
	    if (axis[kw[i].dim-1].type == VELTYPE)
	      axis[kw[i].dim-1].delta /= 1000.0; /* convert m/s to km/s */
	    break;
	  case KW_CROTA:
	    axis[kw[i].dim-1].rotate = kw[i].val.d;
	    break;
	  case KW_BSCALE:
	    bscale = kw[i].val.d;
	    break;
	  case KW_BZERO:
	    bzero  = kw[i].val.d;
	    break;
	  case KW_BUNIT:
	    break;
	  case KW_BLANK:
	    blank = kw[i].val.d;
	    break;
	  case KW_DATAMAX:
	    break;
	  case KW_DATAMIN:
	    break;
	}
    }

#ifdef DEBUG
    for (i = 0; i < naxis; i++) {
	printf("axis #%d:\n", i+1);
	printf("dim:    %d\n", axis[i].dim);
	printf("type:   %d\n", axis[i].type);
	printf("label:  %.16s\n", axis[i].label);
	printf("pixel:  %f\n", axis[i].pixel);
	printf("value:  %f\n", axis[i].value);
	printf("delta:  %f\n", axis[i].delta);
	printf("rotate: %f\n", axis[i].rotate);
    }
    printf("\n");
    printf("bzero:  %f\n", bzero);
    printf("bscale: %f\n", bscale);
#endif
    size = 1;
    for (i = 0; i < naxis; i++) {
	size *= axis[i].dim;
    }
#ifdef DEBUG
    printf("allocating %d floats\n", size);
#endif
    data = calloc(size, sizeof(float));
    if (data == NULL) DRPerror("memory allocation failure");

    free(kw);

    rp = (float *)data;
    switch (bitpix) {
      case 16:
	ip = (WORD *)data;
	i = readFITSdata(fits, size, sizeof(WORD), data);
	if (i == 0) DRPerror("error reading fits data");
	for (i = size-1; i >= 0; i--) {
#ifdef BYTESWAP
        /* The FITS standard expects the binary words with the MSB first */
        /* We have to swap them on e.g. an Intel CPU */
            swapbytes((char *)&ip[i], sizeof(WORD));
#endif
	    if (ip[i] == (WORD)blank) rp[i] = 0.0;
	    else rp[i] = bzero+bscale*ip[i];
	}
#ifdef DEBUG
	for (i = 0; i < axis[0].dim; i++) {
	    for (j = 0; j < axis[1].dim; j++) {
		for (k = 0; k < axis[2].dim; k++) {
		    rp[(k*axis[1].dim + j)*axis[0].dim + i] = j*axis[0].dim+i;
/*		    printf("%f\n", rp[(k*axis[0].dim + j)*axis[1].dim + i]); */
		}
	    }
	}
#endif
	break;
      case 32:
	jp = (LONG *)data;
	i = readFITSdata(fits, size, sizeof(LONG), data);
	if (i == 0) DRPerror("error reading fits data");
	for (i = size-1; i >= 0; i--) {
#ifdef BYTESWAP
        /* The FITS standard expects the binary words with the MSB first */
        /* We have to swap them on e.g. an Intel CPU */
            swapbytes((char *)&jp[i], sizeof(LONG));
#endif
	    if (jp[i] == (LONG)blank) rp[i] = 0.0;
	    else rp[i] = bzero+bscale*jp[i];
	}
	break;
      default:
	DRPerror("can't read BITPIX = %d format", bitpix);
	break;
    }

#ifndef DEBUG    
    if (fork() == 0) {
	strcpy(metaname, getenv("HOME"));
	strcat(metaname, METAFILE);
	if (!overwrite) meta = fopen(metaname, "w");
	else            meta = fopen(metaname, "a");
	if (meta == NULL) DRPerror("can't open metacode file '%s'", metaname);
	plsfile(meta);
	plstart("plmeta", 1, 1); 
    } else {
#endif
	plstart(PLDEVICE, 1, 1); 
	if (!overwrite) plclr();
#ifndef DEBUG    
    }
#endif

    if (!draft) {
	plfontld(1);
	plfont(2);
	plwid(4);
    }

    if (xMin != 0.0 || xMax != 0.0) {
	from = (int)((xMin-axis[v_axis].value)/axis[v_axis].delta 
	     + axis[v_axis].pixel);
	to   = (int)((xMax-axis[v_axis].value)/axis[v_axis].delta 
	     + axis[v_axis].pixel);
	if (from > to) {
	    i = from;
	    from = to;
	    to = i;
	}
	axis[v_axis].pixel = 1.0;
	if (axis[v_axis].delta > 0.0) axis[v_axis].value = xMin;
	else                          axis[v_axis].value = xMax;
	subcube(rp, v_axis, from, to);
    }
    showcube(rp);

    plend();

    free(data);

    exit(EX_SUCCESS);
}
