/***********************************************************************
 *
 * file: drpwindow.c
 * 
 *  The program installs shared memory segment to be used by (most) 
 *  'drp' commands. It also provides an X window for other 'drp' commands
 *  (like 'show') to render their graphics. Once the window is closed
 *  the shared memory segment is removed.
 *
 * who      when         what
 * -------- ------------ -----------------------------------------------
 * olberg   31 Mar 1993  original version
 *
 ***********************************************************************/
#include <X11/Xlib.h>
#include <X11/Xutil.h>

/* We will register our graphics window as a property in the root window
in order for other programs to be able to retrieve that window's id. */
#include <X11/Xatom.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>

#include "drp.h"

#define FALSE 0

/* description of shared memory related stuff */
#include "shmcom.h"
#include "options.h"

/* structure describing shared memory */
static segment *ScanArea;
void shmkill();

#ifdef SHMCOM
int uid;
key_t key;
#endif

char PROGRAM_NAME[] = "drpwindow";
char description[] = "provide shared memory and X11 window";
char required[] = "";
struct _options options[] = {
    { "-help",  "Print out this message" },
    { "-nox",   "don't open X11 window, just provide shared memory" },
    {NULL, NULL }
};

static int x11 = 1;

void ParseOpts(int *pargc, char ***pargv)
{
    char *opt /*, *optarg*/;

    opt = (*pargv)[0] + 1;

    if (!strcmp(opt, "help")) {
        Help();
        exit(EX_SUCCESS);
    }
    if (!strcmp(opt, "nox")) {
        x11 = 0;
        return;
    }
}

/* 
 * Create the shared memory segment and associated semaphore.
 * This will be a 'nop' routine if you have #undef SHMCOM
 */
void shminit(void)
{
#ifdef SHMCOM
    /* Get user ID and form a key by 'or'-ing it with a constant,
       this way we avoid conflicts with other users running DRP at 
       the same time */
    uid = getuid();
    key = (key_t)(L_KEY | uid);

    /* Create a semaphore for locking of our shared memory segment.   */
    /* Create the shared memory segment.                              */
    if ((semid = semget(key, 1, CREATE)) == -1) {
        perror("semget");
        exit(1);
    }
    if ((shmid = shmget(key, sizeof(*ScanArea), CREATE)) == -1) {
        perror("shmget");
        exit(1);
    }

    /* Attach the segment, let the system pick an address.            */
    ScanArea = (segment *)shmat(shmid, (char *)NULL, 0);
    if (ScanArea == (segment *)(-1)) {
        perror("shmat");
        exit(1);
    }

    /* Release the lock semaphore (let others use the memory segment) */
    if (semop(semid, &relse, 1) == -1) {
        perror("semop");
        exit(1);
    }
#endif
}

/* 
 * Close the X window and remove shared memory and semaphore.
 * This function is added as an event handler and will be called
 * on any events. Closing of the window will result in a 'ClientMessage'
 * event, in that case there is no return from this function but an exit
 * of the program. The function will not do anything for all other events.
 */

void shmkill()
{
#ifdef SHMCOM
    int rc1, rc2;

#ifdef DEBUG
    printf("killing shared memory ... ");
#endif

    /* remove shared memory segment */
    /*   rc1 = shmctl(shmid, 0, IPC_RMID); */
    rc1 = shmctl(shmid, IPC_RMID, (struct shmid_ds *)0);
    if (rc1) fprintf (stderr, "rc1 (errno) = %d (%d)\n", rc1, errno);

    /* remove semaphore */
    /*   rc2 = semctl(semid, 0, IPC_RMID, 0); */
    rc2 = semctl(semid, 0, IPC_RMID, 0);
    if (rc2) fprintf (stderr, "rc2 (errno) = %d (%d)\n", rc2, errno);
#ifdef DEBUG
    printf("done\n");
#endif

#endif
}

void SigHandler(int signum)
{
    switch (signum) {
      case SIGQUIT:
        shmkill();
        exit(0);
      default:
        break;
    }
}

XSizeHints hint;
static XEvent event;
XFontStruct *font_info;

typedef struct {
    float xoff;
    float xscl;
    float yoff;
    float yscl;
} World;

int main (int argc, char *argv[])
{
    int x, y;
    int format, n;
    unsigned int width, height, border_width, depth;
    Window window, root;
    Atom DRP_WINDOW, DRP_WORLD, DRP_DIMENSION;
    Atom type, protocols, delete_win;
    Display *dpy;
    /* Status status = 0; */
    GC gc;
    static char header[] = "DRP window";
    static char inq[80];
    unsigned long nitems, left;
    static World world, *retdata;

    GetOpts(argc, argv);

    signal(SIGQUIT, SigHandler);  /* install signal handler */
    shminit();                  /* Install shared memory segment */

    if (x11) {
        dpy = XOpenDisplay(NULL);
        /* status = */ XGetGeometry(dpy, DefaultRootWindow(dpy), &root, 
                                    &x, &y, &width, &height, &border_width, &depth);

        hint.x  = 100;
        hint.y  = 100;
        hint.width  = 2*width/5;
        hint.height = 2*height/5;
        hint.flags  = PPosition|PSize;
        window = XCreateSimpleWindow(dpy, DefaultRootWindow(dpy),
                                     hint.x, hint.y, hint.width, hint.height, 40, 
                                     BlackPixel(dpy, DefaultScreen(dpy)), 
                                     WhitePixel(dpy, DefaultScreen(dpy)));

        XSetStandardProperties(dpy, window, header, header, None, 0, 0, &hint);
        XSelectInput(dpy, window, ExposureMask);
        XSelectInput(dpy, DefaultRootWindow(dpy), SubstructureNotifyMask);

        gc = XCreateGC(dpy, window, 0, 0);
        XSetForeground(dpy, gc, BlackPixel(dpy, DefaultScreen(dpy)));

        DRP_WORLD     = XInternAtom(dpy, "DRP World", 0);
        DRP_DIMENSION = XInternAtom(dpy, "DRP Dimension", 0);
        DRP_WINDOW    = XInternAtom(dpy, "DRP Window", 0);

        XChangeProperty(dpy, DefaultRootWindow(dpy), DRP_WINDOW, XA_WINDOW, 32, 
                        PropModeReplace, (unsigned char *)&window, 1);

        protocols     = XInternAtom(dpy, "WM_PROTOCOLS", 0);
        delete_win    = XInternAtom(dpy, "WM_DELETE_WINDOW", 0);

        /* status = */ XSetWMProtocols(dpy, window, &delete_win, 1);

        XMapRaised(dpy, window);

        while (1) {
            XNextEvent(dpy, &event);
            switch (event.type) {
              case Expose:
                while (XCheckTypedEvent(dpy, Expose, &event)) {
#ifdef DEBUG
                    printf("expose ...\n");
#endif
                }
#ifdef DEBUG
                printf("expose (%d)\n", event.xexpose.count);
#endif
                if (event.xexpose.count == 0) {
                    n = XGetWindowProperty(dpy, DefaultRootWindow(dpy), DRP_WORLD, 
                                           0, sizeof(World)/4, FALSE, 
                                           DRP_DIMENSION, 
                                           &type, &format, &nitems, &left, 
                                           (unsigned char **)&retdata);
                    if (n == Success && type == DRP_DIMENSION) {
                        world.xoff = retdata->xoff;
                        world.xscl = retdata->xscl;
                        world.yoff = retdata->yoff;
                        world.yscl = retdata->yscl;
                    }
                    system("reshow");
                    XChangeProperty(dpy, DefaultRootWindow(dpy), 
                                    DRP_WORLD, DRP_DIMENSION, 32, 
                                    PropModeReplace, (unsigned char *)&world, 
                                    sizeof(World)/4);
                }
                break;
              case ConfigureNotify:
#ifdef DEBUG
                printf("configure\n");
#endif
                break;
              case DestroyNotify:
#ifdef DEBUG
                printf("destroy\n");
#endif
                break;
              case ButtonPress:
#ifdef DEBUG
                printf("button\n");
#endif
                n = XGetWindowProperty(dpy, DefaultRootWindow(dpy), DRP_WORLD, 
                                       0, sizeof(World)/4, FALSE, DRP_DIMENSION, 
                                       &type, &format, &nitems, &left, 
                                       (unsigned char **)&retdata);
                if (n != Success || type != DRP_DIMENSION) {
                    fprintf(stderr, "can't retrieve window properties\n");
                    exit(1);
                }
                world.xoff = retdata->xoff;
                world.xscl = retdata->xscl;
                world.yoff = retdata->yoff;
                world.yscl = retdata->yscl;
                /* status = */ XGetGeometry(dpy, window, &root, 
                                            &x, &y, &width, &height, 
                                            &border_width, &depth);
                sprintf(inq,"%10.3f %10.3f", 
                        world.xoff+event.xbutton.x*2*world.xscl,
                        world.yoff+(height-event.xbutton.y)*2*world.yscl);
                x = 2;
                y = 20;
                XDrawImageString(dpy, window, gc, x, y, inq, strlen(inq));
                break;
              case KeyPress:
#ifdef DEBUG
                printf("key pressed\n");
#endif
                break;
              case EnterNotify:
#ifdef DEBUG
                printf("enter\n");
#endif
                break;
              case LeaveNotify:
#ifdef DEBUG
                printf("leave\n");
#endif
                break;
              case ClientMessage:
#ifdef DEBUG
                printf("Client message\n");
                printf("type = %lx\n", event.xclient.message_type);
                printf("format = %d\n", event.xclient.format);
                for (i=0; i<5; i++) 
                    printf("data[%d] = %lx\n", i, event.xclient.data.l[i]);
#endif
                if(event.xclient.message_type == protocols) {
                    if (event.xclient.data.l[0] == delete_win) {
#ifdef DEBUG
                        printf("delete window\n");
#endif
                        shmkill();
                        XCloseDisplay(dpy);
                        exit(0);
                    }
                }
                break;
              default:
#ifdef DEBUG
                printf("other (%x)\n", event.type);
#endif
                break;
            }
        }
    } else {
        pause();
    }
    return 0;
}
