#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "drp.h"
#include "options.h"

#define MAXDIM 30
 
static int adim;
static int order;
static double a[MAXDIM];

void savgol(double [], int , int , int , int , int );

char PROGRAM_NAME[] = "filter";
char description[] = "apply digital filter to spectrum";
char required[] = "";
struct _options options[] = {
{ "-help",	       "Print out this message" },
{ "-boxcar n",         "apply (2*n-1)-channel boxcar filter" },
{ "-hanning",          "apply hanning (1/4 1/2 1/4) filter" },
{ "-dispo m n",        "apply m-th order, n-channel least-squares filter" },
{ "-coeff n a[0]...a[n-1]",  "specify your own filter (one half)" },
{NULL, NULL }};

void ParseOpts(int *pargc, char ***pargv)
{
    char *opt, *optarg;
    int i;

    opt = (*pargv)[0] + 1;

    if (!strcmp(opt, "help")) {
	Help();
	exit(0);
    }
    if (!strcmp(opt, "boxcar")) {
	GetOption(&optarg, pargc, pargv);
	adim = atoi(optarg);
	if (adim > MAXDIM) DRPerror("maximum number of coefficients exceeded");
	for (i = 0; i < adim; i++) {
	    a[i] = 1.0;
	}
	return;
    }
    if (!strcmp(opt, "hanning")) {
	GetOption(&optarg, pargc, pargv);
	adim = 2;
	a[0] = 0.5;
	a[1] = 0.25;
	return;
    }
    if (!strcmp(opt, "dispo")) {
	GetOption(&optarg, pargc, pargv);
	order = atoi(optarg);
	GetOption(&optarg, pargc, pargv);
	adim = atoi(optarg);
	if (adim > MAXDIM) DRPerror("maximum number of coefficients exceeded");
	savgol(a, 2*adim+1, adim, adim, 0, order);
	adim++;
	return;
    }
    if (!strcmp(opt, "coeff")) {
	GetOption(&optarg, pargc, pargv);
	adim = atoi(optarg);
	if (adim > MAXDIM) DRPerror("maximum number of coefficients exceeded");
	for (i = 0; i < adim; i++) {
	    GetOption(&optarg, pargc, pargv);
	    a[i] = atof(optarg);
	}
	return;
    }
    Syntax(**pargv);
}

void twofft(float [], float [], float [], float [], int );
void realft(float [], int , int);

#define min(a,b) ((a) < (b) ? (a) : (b))

int ludcmp(double **, int , int [], double *);
void lubksb(double **, int , int [], double []);

void savgol(double c[], int np, int nl, int nr, int ld, int m)
{
    int i, imj, ipj, j, k, kk, mm, *index;
    double d, fac, sum, **A, *b;

    A = (double **)calloc(m+1, sizeof(double *));
    if (A == NULL) DRPerror("memory allocation failure");

    for (i = 0; i < m+1; i++) {
        A[i] = (double *)calloc(m+1, sizeof(double));
	if (A[i] == NULL) DRPerror("memory allocation failure");
    }
    b = (double *)calloc(m+1, sizeof(double));
    if (b == NULL) DRPerror("memory allocation failure");

    index = (int *)calloc(m+1, sizeof(int));
    if (index == NULL) DRPerror("memory allocation failure");

    for (ipj = 0; ipj <= 2*m; ipj++) {
        sum = 0.0;
        if (ipj == 0) sum = 1.0;
        for (k = 0; k < nl; k++) sum += pow((double)( k+1),(double)ipj);
        for (k = 0; k < nr; k++) sum += pow((double)(-k-1),(double)ipj);
        mm = min(ipj, 2*m-ipj);
        for (imj = -mm; imj <= mm; imj += 2) {
            A[(ipj-imj)/2][(ipj+imj)/2] = sum;
	}
    }

    if (ludcmp(A, m+1, index, &d) != 0) DRPerror("matrix singular");

    for (j = 0; j < m+1; j++) b[j] = 0.0;
    b[ld] = 1.0;
    lubksb(A, m+1, index, b);

    for (kk = 0; kk < np; kk++) c[kk] = 0.0;
    for (k = -nl; k <= nr; k++) {
        sum = b[0];
        fac = 1.0;
        for (mm = 1; mm <= m; mm++) {
            fac *= (double)k;
            sum += b[mm]*fac;
	}
        kk = (np-k) % np;
        c[kk] = sum;
    }

    free(index);
    free(b);
    for (i = 0; i < m+1; i++) free(A[i]);
    free(A);
}

SCAN OnScan, temp;

int main(int argc, char *argv[])
{
    int  i, j, m, n;
    float *fft1, *fft2;
    float re, im;
    double norm;

    a[0] = 1.0;
    GetOpts(argc, argv);

    if (adim == 0) DRPerror("filter coefficients required");

    norm = a[0];
    for (j = 1; j < adim; j++) norm += 2.0*a[j];
    for (j = 0; j < adim; j++) a[j] /= norm;
/*  for (j = 0; j < adim; j++) printf("a[%d] = %lf\n", j, a[j]); */

    GetScan(&OnScan);
    memcpy(&temp, &OnScan, sizeof(SCAN));
    
    for (n = 2; n < OnScan.NChannel; n *= 2) {}

    temp.Channels[0] = a[0];
    for (i = 1; i < adim; i++) temp.Channels[i] = temp.Channels[n-i] = a[i];
    for (i = adim; i <= n-adim; i++) temp.Channels[i] = 0.0;

    fft1 = (float *)calloc(2*n, sizeof(float));
    if (fft1 == NULL) DRPinfo("memory allocation failure");

    fft2 = (float *)calloc(2*n, sizeof(float));
    if (fft2 == NULL) DRPinfo("memory allocation failure");

    twofft(OnScan.Channels, temp.Channels, fft1, fft2, n);

    m = n/2;
    for (i = 0; i <= n; i += 2) {
	re = fft1[i]*fft2[i] - fft1[i+1]*fft2[i+1];
	im = fft1[i]*fft2[i+1] + fft1[i+1]*fft2[i];
	fft1[i]   = re/m;
	fft1[i+1] = im/m;
    }
    fft1[1] = fft1[n];

    realft(fft1, m, -1);
    for (i = 0; i < OnScan.NChannel; i++) temp.Channels[i] = fft1[i];

    free(fft1);
    free(fft2);

    PutScan(&temp);

    exit(0);
}
