#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <time.h>
#include <ctype.h>
#include "drp.h"
#include "options.h"
#include "fits.h"
  
static char cubename[MAXNAMLEN+1];
static char nextscan[MAXNAMLEN+1];

static int col = 1;

#define READ 0

#define EQUATORIAL 1
#define GALACTIC   5
static int csys = EQUATORIAL;
static int magnify = 1;
static int nsmooth = 0;

#define MAXLEN 200
#define MAXCOL  10
#define MAXDIM 2000

void scanio(char *, int , SCAN *);

/* static struct point { */
/*     int mapx, mapy; */
/*     int scanno; */
/* } *p; */

static struct point {
    int mapx, mapy;
    int scanno;
} p[MAXDIM];

char PROGRAM_NAME[] = "fitscube";
char description[] = "store scans from log-file as 3-dim (X x Y x Vel) FITS cube";
char required[] = "";
struct _options options[] = {
{ "-help",		"Print out this message" },
{ "-file filename",     "specify output filename (default: SOURCENAME.CUBE" },
{ "-magnify m",         "fill in m addititional pixels between each grid positions" },
{ "-smooth n",          "smooth each plane of cube n times" },
{NULL, NULL }};

void ParseOpts(int *pargc, char ***pargv)
{
  char *opt, *optarg;

  opt = (*pargv)[0] + 1;

  if (!strcmp(opt, "help")) {
    Help();
    exit(0);
  }

  /* file name */
  if (!strcmp(opt, "file")) {
    GetOption(&optarg, pargc, pargv);
    strcpy(cubename, optarg);
    return;
  }
  if (!strcmp(opt, "col")) {
    GetOption(&optarg, pargc, pargv);
    col = atoi(optarg);
    if (col < 1 || col > MAXCOL) Syntax("-col");
    return;
  }
  if (!strcmp(opt, "magnify")) {
    GetOption(&optarg, pargc, pargv);
    magnify = atoi(optarg);
    if (magnify < 1) Syntax("-magnify");
    return;
  }
  if (!strcmp(opt, "smooth")) {
    GetOption(&optarg, pargc, pargv);
    nsmooth = atoi(optarg);
    if (nsmooth < 1) Syntax("-smooth");
    return;
  }

  Syntax(**pargv);
}

#define BLANK (-32768)

typedef short int WORD;

#define FALSE      0
#define TRUE       1
  
struct fitskey card;
  
void pixels(float ,float ,float ,float ,float ,int *,int *);
int scanlog(void);
void KillJunk(char []);  

extern int used;
extern struct fitskey *kw;
struct fitskey card;
  
void addFITScard(struct fitskey *);
int writeFITSheader(FILE *);
int writeFITSdata(FILE *, int ,int ,void *);
#ifdef BYTESWAP
void swapbytes(char *, int );
#endif

void VoidCard(int index)
{
  card.key = index;
  card.dim = 0;
  addFITScard(&card);
}

void BoolCard(int index, int b)
{
  card.key = index;
  card.dim = 0;
  card.val.l = (long)b;
  addFITScard(&card);
}

void CharCard(int index, int dim, char *s)
{
  int i, len;

  card.key = index;
  card.dim = dim;
  strncpy(card.val.str, s, 16);
  len = strlen(card.val.str);
  if (len < 8) {
    for (i = len; i < 8; i++) card.val.str[i] = ' ';
    card.val.str[8] = '\0';
  }
  addFITScard(&card);
}

void LongCard(int index, int dim, long l)
{
  card.key = index;
  card.dim = dim;
  card.val.l = l;
  addFITScard(&card);
}

void RealCard(int index, int dim, double d)
{
  card.key = index;
  card.dim = dim;
  card.val.d = d;
  addFITScard(&card);
}

static WORD *cube;
static float *spectra;
static float zmin, zmax, bscale, bzero;
static int xminpix, xmaxpix, yminpix, ymaxpix, DimX, DimY, DimZ;

SCAN OnScan, temp;

void pixels(float xoff, float yoff, float stepx, float stepy, float tilt,
	int *ipix, int *jpix)
{
  *ipix = drpint((xoff*cos(tilt)-yoff*sin(tilt))/stepx);
  *jpix = drpint((xoff*sin(tilt)+yoff*cos(tilt))/stepy);
}
  
int scanlog(void)
{
  FILE *drplog;
  char record[MAXLEN], *com, *next;
  float ox = 0.0, oy = 0.0;
  float tilt,stepx,stepy;
  int pts = 0;
  int no = 0;
 
  stepx = OnScan.StepX; stepx *= RADTOMIN;
  stepy = OnScan.StepY; stepy *= RADTOMIN;
  tilt = OnScan.PosAngle;
  if (stepx == 0.0 || stepy == 0.0) DRPerror("zero map spacing");
  
  drplog = fopen("drp.log", "r");
  if (drplog == NULL) DRPerror("couldn't open log file");

  /*
   * scan the log file (DRP.LOG) for entries and store them by writing
   * over the channel values in OnScan.
   */

  pts = 0;			/* count the entries */
  while (fgets(record, MAXLEN, drplog)) {
    com = strtok(record, ":");
    if (com == NULL) DRPerror("syntax error in file 'drp.log'");

    next = strtok(NULL,"\t\n\0");
    if (next == NULL) DRPerror("syntax error in file 'drp.log'");
    else no = atoi(next);

    next = strtok(NULL,"\t\n\0");
    if (next == NULL) DRPerror("syntax error in file 'drp.log'");
    else ox = atof(next);

    next = strtok(NULL,"\t\n\0");
    if (next == NULL) DRPerror("syntax error in file 'drp.log'");
    else oy = atof(next);

    if (com) {
      pixels(ox, oy, stepx, stepy, tilt, 
	     &(p[pts].mapx), &(p[pts].mapy));
      p[pts].scanno = no;
      if (pts == 0) {
	xminpix = xmaxpix = p[pts].mapx;
	yminpix = ymaxpix = p[pts].mapy;
      } else {
	if (xminpix > p[pts].mapx) xminpix = p[pts].mapx;
	if (xmaxpix < p[pts].mapx) xmaxpix = p[pts].mapx;
	if (yminpix > p[pts].mapy) yminpix = p[pts].mapy;
	if (ymaxpix < p[pts].mapy) ymaxpix = p[pts].mapy;
      }
      pts++;
      if (pts > MAXDIM) DRPerror("too many entries");
    }
  }

  fclose(drplog);
  if (pts == 0) DRPerror("no data found");
  return (pts);
}

void KillJunk(char s[])
{
  int i,j;

  for (i=0, j=0; i<strlen(s); i++)
    if (isalnum(s[i])) s[j++] = s[i];
  s[j++] = '\0';
}

void smooth(WORD *image, int nx, int ny)
{
  WORD *tmp;
  int i, j, m, pixel, l;

  tmp = (WORD *)calloc(nx*ny, sizeof(WORD));
  if (tmp == (WORD *)NULL) DRPerror("can't allocate temporary map");

  for (l = 0; l < nsmooth; l++) {
    memcpy(tmp, image, nx*ny*sizeof(WORD));

    for (j = 1; j < ny-1; j++) {
      for (i = 1; i < nx-1; i++) {
	pixel = j*nx+i;
	if (tmp[pixel-   1] != BLANK && tmp[pixel+   1] != BLANK
	    && tmp[pixel-nx] != BLANK && tmp[pixel+nx] != BLANK) {
	  m = tmp[pixel-1] + tmp[pixel+1] + tmp[pixel-nx] + tmp[pixel+nx];
	  image[pixel] = m/4;
	}
      }
    }
  }
  free(tmp);
}

int main(int argc, char *argv[])
{
  FILE *fits;
  int i, j, m, n, c, pts, pixel;
  WORD *plane;
  int diff;
  double v, dv, f, df;
  char string[20], ext[5];
  struct tm *now;
  time_t tick;

  GetOpts(argc, argv);

  GetScan(&OnScan);
  csys = OnScan.CSystem;
  if (OnScan.StepX > 0.0) OnScan.StepX = -OnScan.StepX;

  if (cubename[0] == '\0') {
    strncpy(cubename, OnScan.Name, 12);
    KillJunk(cubename);
    strcat(cubename, ".CUBE");
  }
  fits = fopen(cubename, "w");        /* open the file for reading  */
  if (fits == NULL) DRPerror("can't open file '%s'", cubename);

  pts = scanlog();
  DimX = (xmaxpix-xminpix)*magnify+1;
  DimY = (ymaxpix-yminpix)*magnify+1;
  DimZ = OnScan.NChannel;
  v  = OnScan.VSource;
  dv = OnScan.VelRes;
  f  = OnScan.RestFreq;
  df = OnScan.FreqRes;

  spectra = (float *)calloc(pts*DimZ, sizeof(float));
  if (spectra == NULL) DRPerror("can't allocate memory for spectra");
  
  zmin = zmax = 0.0;
  for (m = 0; m < pts; m++) {
    n = p[m].scanno;
    memset(nextscan, '\0', MAXNAMLEN+1);
    strncpy(nextscan, OnScan.Name, 12);
    KillJunk(nextscan);
    sprintf (ext, ".%04d", p[m].scanno);
    strcat(nextscan, ext);
    DRPinfo("reading scan '%s'", nextscan);
    scanio(nextscan, READ, &temp);
    if (temp.NChannel != DimZ) DRPerror("number of channel mismatch");
    if (fabs(temp.VSource-v) > fabs(dv))
      DRPwarning("source velocity mismatch: %lf %lf", temp.VSource, v);
    if (fabs((temp.VelRes - dv)/dv) > 0.01)
      DRPwarning("velocity resolution mismatch: %lf %lf", temp.VelRes, dv);
    if (fabs(temp.RestFreq - f) > fabs(df))
      DRPwarning("rest frequency mismatch: %lf %lf", temp.RestFreq, f);
    if (fabs((temp.FreqRes - df)/df) > 0.01)
      DRPwarning("frequency resolution mismatch: %lf %lf", temp.FreqRes, df);
    for (c = 0; c < DimZ; c++) {
      if (zmin > temp.Channels[c]) zmin = temp.Channels[c]; 
      if (zmax < temp.Channels[c]) zmax = temp.Channels[c];
      spectra[m*DimZ+c] = temp.Channels[c];
    }
  }

  used = 0;

  bzero = 0.5*(zmax+zmin);
  bscale = (zmax-bzero)/32767;
 
  BoolCard(KW_SIMPLE, TRUE);
  LongCard(KW_BITPIX, 0, 16);
  LongCard(KW_NAXIS, 0, 3);
  LongCard(KW_NAXIS, 1, (long)DimX);
  LongCard(KW_NAXIS, 2, (long)DimY);
  LongCard(KW_NAXIS, 3, (long)DimZ);
  /*
    LongCard(KW_NAXIS, 4, 1);
    */    
  switch (csys) {
   case EQUATORIAL:
    CharCard(KW_CTYPE, 1, "RA---SIN");
    break;
   case GALACTIC:
    CharCard(KW_CTYPE, 1, "GLON-SIN");
    break;
   default:
    CharCard(KW_CTYPE, 1, "RA---SIN");
    break;
  }
  RealCard(KW_CRPIX, 1, (double)(1-xminpix*magnify));
  RealCard(KW_CRVAL, 1, (double)OnScan.Longitude*RADTODEG);
  RealCard(KW_CDELT, 1, (double)OnScan.StepX*RADTODEG/magnify);
  RealCard(KW_CROTA, 1, (double)OnScan.PosAngle*RADTODEG);

  switch (csys) {
   case EQUATORIAL:
    CharCard(KW_CTYPE, 2, "DEC--SIN");
    break;
   case GALACTIC:
    CharCard(KW_CTYPE, 2, "GLAT-SIN");
    break;
   default:
    CharCard(KW_CTYPE, 2, "DEC--SIN");
    break;
  }
  RealCard(KW_CRPIX, 2, (double)(1-yminpix*magnify));
  RealCard(KW_CRVAL, 2, (double)OnScan.Latitude*RADTODEG);
  RealCard(KW_CDELT, 2, (double)OnScan.StepY*RADTODEG/magnify);
  RealCard(KW_CROTA, 2, (double)OnScan.PosAngle*RADTODEG);

  CharCard(KW_CTYPE, 3, "VELO-LSR");
  RealCard(KW_CRPIX, 3, (double)(CenterCh(&OnScan)+1));
  RealCard(KW_CRVAL, 3, v*1.0e3);
  RealCard(KW_CDELT, 3, dv*1.0e3);
  /*
    CharCard(KW_CTYPE, 4, "FREQ");
    RealCard(KW_CRPIX, 4, 1.0);
    RealCard(KW_CRVAL, 4, f*1.0e6);
    RealCard(KW_CDELT, 4, df*1.0e6);
    */
  CharCard(KW_BUNIT, 0, "K");
  RealCard(KW_BSCALE, 0, (double)bscale);
  RealCard(KW_BZERO, 0, (double)bzero);
  RealCard(KW_DATAMAX, 0, (double)zmax);
  RealCard(KW_DATAMIN, 0, (double)zmin);
  RealCard(KW_BLANK, 0, (double)BLANK);
  RealCard(KW_RESTFREQ, 0, f*1.0e6);

  /* sprintf(string, "%02d/%02d/%02d", 
	  OnScan.Day, OnScan.Month, OnScan.Year%100); */
  sprintf(string, "%04d-%02d-%02d", 
	  OnScan.Year, OnScan.Month, OnScan.Day);
  CharCard(KW_DATE_OBS, 0, string);
  time(&tick);
  now = localtime(&tick);
  /* sprintf(string,"%02d/%02d/%02d",
	  now->tm_mday, now->tm_mon+1, now->tm_year%100); */
  sprintf(string, "%04d-%02d-%02d", 
	  now->tm_year+1900, now->tm_mon+1, now->tm_mday);
  CharCard(KW_DATE, 0, string);
  strncpy(string, OnScan.Name, 12); string[12] = '\0';
  CharCard(KW_OBJECT, 0, string);
  CharCard(KW_ORIGIN, 0, "DRP");
  if (csys == EQUATORIAL) RealCard(KW_EQUINOX, 0, (double)OnScan.Equinox);
  strncpy(string, OnScan.Observer, 16); string[16] = '\0';
  CharCard(KW_OBSERVER, 0, string);
  VoidCard(KW_END);

  writeFITSheader(fits);
  free(kw);

  cube = (WORD *)calloc(DimX*DimY*DimZ, sizeof(WORD));
  if (cube == NULL) DRPerror("can't allocate cube");
  
  for (n = 0; n < DimX*DimY*DimZ; n++) cube[n] = BLANK;

  plane = cube;
  for (c = 0; c < DimZ; c++) {
    DRPinfo("processing plane %d", c);
/*     p = (struct point *)OnScan.Channels; */
    for (n = 0; n < pts; n++) {
      i = p[n].mapx; 
      j = p[n].mapy;
      pixel = ((j-yminpix)*DimX+(i-xminpix))*magnify;
      plane[pixel] = (WORD)((spectra[n*DimZ+c]-bzero)/bscale);
    }

    if (magnify > 1) {
      for (j = 0; j < DimY; j += magnify) {
	for (i = magnify; i < DimX; i += magnify) {
	  pixel = j*DimX+i-magnify;
	  if (plane[pixel] != BLANK && plane[pixel+magnify] != BLANK) {
	    diff = plane[pixel+magnify] - plane[pixel];
	    for (m = 1; m < magnify; m++) {
	      plane[pixel+m] = plane[pixel] + (WORD)(m*diff/magnify);
	    }
	  }
	}
      }
      for (i = 0; i < DimX; i++) {
	for (j = magnify; j < DimY; j += magnify) {
	  pixel = (j-magnify)*DimX+i;
	  if (plane[pixel] != BLANK && plane[pixel+magnify*DimX] != BLANK) {
	    diff = plane[pixel+magnify*DimX] - plane[pixel];
	    for (m = 1; m < magnify; m++) {
	      plane[pixel+m*DimX] = plane[pixel] + (WORD)(m*diff/magnify);
	    } 
	  }
	}
      }
      if (nsmooth) {
	smooth(plane, DimX, DimY);
      }
    }
    plane += DimX*DimY;
  }

#ifdef BYTESWAP
  /* The FITS standard expects the binary words with the MSB first */
  /* We have to swap them on e.g. an Intel CPU */
  for (i = 0; i < DimX*DimY*DimZ; i++) {
    swapbytes((char *)&cube[i], sizeof(WORD));
  }
#endif

  writeFITSdata(fits, DimX*DimY*DimZ, sizeof(WORD), (void *)cube);
  free(cube);
    
  DRPinfo("cube written to file '%s'", cubename);

  exit(0);
}
