#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <time.h>
#include <ctype.h>
#include "drp.h"
#include "options.h"
#include "fits.h"
  
static char scanname[MAXNAMLEN+1];

static int col = 1;

#define EQUATORIAL 1
#define GALACTIC   5

static int csys = EQUATORIAL;
static int magnify = 1;
static int nsmooth = 0;

#define MAXLEN 200
#define MAXCOL  10
#define MAXDIM 2000
  
char PROGRAM_NAME[] = "fitsmap";
char description[] = "store log results as 2-dim FITS file";
char required[] = "";
struct _options options[] = {
{ "-help",		"Print out this message" },
{ "-file filename",     "specify output filename (default: SOURCENAME.MAP" },
{ "-col number",        "specify column number to map" },
{ "-magnify m",         "fill in m addititional pixels between each grid positions" },
{ "-smooth n",          "smooth map n times" },
/* { "-system name",       "specify coordinate system (EQUATORIAL|GALACTIC)" }, */
{NULL, NULL }};

void ParseOpts(int *pargc, char ***pargv)
{
  char *opt, *optarg;

  opt = (*pargv)[0] + 1;

  if (!strcmp(opt, "help")) {
    Help();
    exit(0);
  }

  /* file name */
  if (!strcmp(opt, "file")) {
    GetOption(&optarg, pargc, pargv);
    strcpy(scanname, optarg);
    return;
  }
  if (!strcmp(opt, "col")) {
    GetOption(&optarg, pargc, pargv);
    col = atoi(optarg);
    if (col < 1 || col > MAXCOL) Syntax("-col");
    return;
  }
  if (!strcmp(opt, "magnify")) {
    GetOption(&optarg, pargc, pargv);
    magnify = atoi(optarg);
    if (magnify < 1) Syntax("-magnify");
    return;
  }
  if (!strcmp(opt, "smooth")) {
    GetOption(&optarg, pargc, pargv);
    nsmooth = atoi(optarg);
    if (nsmooth < 1) Syntax("-smooth");
    return;
  }

  Syntax(**pargv);
}

#define BLANK (-32768)

typedef short int WORD;

#define FALSE      0
#define TRUE       1
  
extern int used;
extern struct fitskey *kw;
struct fitskey card;
  
void pixels(float ,float ,float ,float ,float ,int *,int *);
int scanlog(void);
void KillJunk(char []);

void addFITScard(struct fitskey *);
int writeFITSheader(FILE *);
int writeFITSdata(FILE *, int ,int ,void *);
#ifdef BYTESWAP
void swapbytes(char *, int );
#endif

void VoidCard(int index)
{
  card.key = index;
  card.dim = 0;
  addFITScard(&card);
}

void BoolCard(int index, int b)
{
  card.key = index;
  card.dim = 0;
  card.val.l = (long)b;
  addFITScard(&card);
}

void CharCard(int index, int dim, char *s)
{
  int i, len;

  card.key = index;
  card.dim = dim;
  strncpy(card.val.str, s, 16);
  len = strlen(card.val.str);
  if (len < 8) {
    for (i = len; i < 8; i++) card.val.str[i] = ' ';
    card.val.str[8] = '\0';
  }
  addFITScard(&card);
}

void LongCard(int index, int dim, long l)
{
  card.key = index;
  card.dim = dim;
  card.val.l = l;
  addFITScard(&card);
}

void RealCard(int index, int dim, double d)
{
  card.key = index;
  card.dim = dim;
  card.val.d = d;
  addFITScard(&card);
}

static WORD *map;
static float zmin, zmax, bscale, bzero;
static int xminpix, xmaxpix, yminpix, ymaxpix, DimX, DimY;

SCAN OnScan;

static struct point {
    int mapx, mapy;
    float value;
} p[MAXDIM];

void pixels(float xoff, float yoff, float stepx, float stepy, float tilt,
	int *ipix, int *jpix)
{
  *ipix = drpint((xoff*cos(tilt)-yoff*sin(tilt))/stepx);
  *jpix = drpint((xoff*sin(tilt)+yoff*cos(tilt))/stepy);
}
  
int scanlog(void)
{
  FILE *drplog;
  char record[MAXLEN], *com, *next;
  float ox = 0.0, oy = 0.0, v[MAXCOL];
  float tilt,stepx,stepy;
  int pts = 0;
  int i, n;
 
  stepx = OnScan.StepX; stepx *= RADTOMIN;
  stepy = OnScan.StepY; stepy *= RADTOMIN;
  tilt = OnScan.PosAngle;
  if (stepx == 0.0 || stepy == 0.0) DRPerror("zero map spacing");
  
  drplog = fopen("drp.log", "r");
  if (drplog == NULL) DRPerror("couldn't open log file");

  /*
   * scan the log file (DRP.LOG) for entries and store them by writing
   * over the channel values in OnScan.
   */
  n = col;
  /* p = (struct point *)OnScan.Channels; */
  pts = 0;			/* count the entries */
  while (fgets(record, MAXLEN, drplog)) {
    com = strtok(record, ":");
    if (com == NULL) DRPerror("syntax error in file 'drp.log'");

    next = strtok(NULL,"\t\n\0");
    if (next == NULL) DRPerror("syntax error in file 'drp.log'");

    next = strtok(NULL,"\t\n\0");
    if (next == NULL) DRPerror("syntax error in file 'drp.log'");
    else ox = atof(next);

    next = strtok(NULL,"\t\n\0");
    if (next == NULL) DRPerror("syntax error in file 'drp.log'");
    else oy = atof(next);

    if (com) {
      for (i=0; i<MAXCOL; i++) {
	next = strtok(NULL,"\t\n\0");
	if (next) v[i] = atof(next);
	else break;
      }
      pixels(ox, oy, stepx, stepy, tilt, 
	     &(p[pts].mapx), &(p[pts].mapy));
      p[pts].value = v[n-1];
      printf("value = %f\n", p[pts].value);
      if (pts == 0) {
	xminpix = xmaxpix = p[pts].mapx;
	yminpix = ymaxpix = p[pts].mapy;
	zmin = zmax = p[pts].value;
      } else {
	if (xminpix > p[pts].mapx) xminpix = p[pts].mapx;
	if (xmaxpix < p[pts].mapx) xmaxpix = p[pts].mapx;
	if (yminpix > p[pts].mapy) yminpix = p[pts].mapy;
	if (ymaxpix < p[pts].mapy) ymaxpix = p[pts].mapy;
	if (zmin > p[pts].value) zmin = p[pts].value;
	if (zmax < p[pts].value) zmax = p[pts].value;
      }
      pts++;
      if (pts > MAXDIM) DRPerror("too many entries");
    }
  }
  bzero = 0.5*(zmax+zmin);
  bscale = (zmax-bzero)/32767;
 
  fclose(drplog);
  if (pts == 0) DRPerror("no data found");
  return (pts);
}

void KillJunk(char s[])
{
  int i,j;

  for (i=0, j=0; i<strlen(s); i++)
    if (isalnum(s[i])) s[j++] = s[i];
  s[j++] = '\0';
}

void smooth(WORD *image, int nx, int ny)
{
  WORD *tmp;
  int i, j, m, pixel, l;

  tmp = (WORD *)calloc(nx*ny, sizeof(WORD));
  if (tmp == (WORD *)NULL) DRPerror("can't allocate temporary map");

  for (l = 0; l < nsmooth; l++) {
    memcpy(tmp, image, nx*ny*sizeof(WORD));

    for (j = 1; j < ny-1; j++) {
      for (i = 1; i < nx-1; i++) {
	pixel = j*nx+i;
	if (tmp[pixel- 1] != BLANK && tmp[pixel+ 1] != BLANK
	    && tmp[pixel-nx] != BLANK && tmp[pixel+nx] != BLANK) {
	  m = tmp[pixel-1] + tmp[pixel+1] + tmp[pixel-nx] + tmp[pixel+nx];
	  image[pixel] = m/4;
	}
      }
    }
  }
  free(tmp);
}

int main(int argc, char *argv[])
{
  FILE *fits;
  int i, j, m, n, pts, pixel;
  int diff;
  char string[20];
  struct tm *now;
  time_t tick;

  GetOpts(argc, argv);
  GetScan(&OnScan);
  csys = OnScan.CSystem;
  if (OnScan.StepX > 0.0) OnScan.StepX = -OnScan.StepX;

  if (scanname[0] == '\0') {
    strncpy(scanname, OnScan.Name, 12);
    KillJunk(scanname);
    strcat(scanname, ".MAP");
  }
  fits = fopen(scanname, "w");        /* open the file for reading  */
  if (fits == NULL) DRPerror("can't open file '%s'", scanname);
  used = 0;

  pts = scanlog();
  DimX = (xmaxpix-xminpix)*magnify+1;
  DimY = (ymaxpix-yminpix)*magnify+1;

  BoolCard(KW_SIMPLE, TRUE);
  LongCard(KW_BITPIX, 0, 16);
  LongCard(KW_NAXIS, 0, 2);
  LongCard(KW_NAXIS, 1, (long)DimX);
  LongCard(KW_NAXIS, 2, (long)DimY);
  /*
    LongCard(KW_NAXIS, 3, 1);
    LongCard(KW_NAXIS, 4, 1);
    */    
  switch (csys) {
   case EQUATORIAL:
    CharCard(KW_CTYPE, 1, "RA---SIN");
    break;
   case GALACTIC:
    CharCard(KW_CTYPE, 1, "GLON-SIN");
    break;
   default:
    CharCard(KW_CTYPE, 1, "RA---SIN");
    break;
  }
  RealCard(KW_CRPIX, 1, (double)(1-xminpix*magnify));
  RealCard(KW_CRVAL, 1, (double)OnScan.Longitude*RADTODEG);
  RealCard(KW_CDELT, 1, (double)OnScan.StepX*RADTODEG/magnify);
  RealCard(KW_CROTA, 1, (double)OnScan.PosAngle*RADTODEG);

  switch (csys) {
   case EQUATORIAL:
    CharCard(KW_CTYPE, 2, "DEC--SIN");
    break;
   case GALACTIC:
    CharCard(KW_CTYPE, 2, "GLAT-SIN");
    break;
   default:
    CharCard(KW_CTYPE, 2, "DEC--SIN");
    break;
  }
  RealCard(KW_CRPIX, 2, (double)(1-yminpix*magnify));
  RealCard(KW_CRVAL, 2, (double)OnScan.Latitude*RADTODEG);
  RealCard(KW_CDELT, 2, (double)OnScan.StepY*RADTODEG/magnify);
  RealCard(KW_CROTA, 2, (double)OnScan.PosAngle*RADTODEG);
  /*
    CharCard(KW_CTYPE, 3, "VELO-LSR");
    RealCard(KW_CRPIX, 3, 1.0);

    CharCard(KW_CTYPE, 4, "FREQ");
    RealCard(KW_CRPIX, 4, 1.0);
    */
  CharCard(KW_BUNIT, 0, "K");
  RealCard(KW_BSCALE, 0, (double)bscale);
  RealCard(KW_BZERO, 0, (double)bzero);
  RealCard(KW_DATAMAX, 0, (double)zmax);
  RealCard(KW_DATAMIN, 0, (double)zmin);
  RealCard(KW_BLANK, 0, (double)BLANK);
  RealCard(KW_VELO_LSR, 0, OnScan.VSource*1.0e3);
  RealCard(KW_DELTAV, 0, OnScan.VelRes*1.0e3);
  RealCard(KW_RESTFREQ, 0, OnScan.RestFreq*1.0e6);

  /* sprintf(string, "%02d/%02d/%02d", 
	  OnScan.Day, OnScan.Month, OnScan.Year%100); */
  sprintf(string, "%04d-%02d-%02d", 
	  OnScan.Year, OnScan.Month, OnScan.Day);
  CharCard(KW_DATE_OBS, 0, string);
  time(&tick);
  now = localtime(&tick);
  /* sprintf(string,"%02d/%02d/%02d",
	  now->tm_mday, now->tm_mon+1, now->tm_year%100); */
  sprintf(string, "%04d-%02d-%02d", 
	  now->tm_year+1900, now->tm_mon+1, now->tm_mday);
  CharCard(KW_DATE, 0, string);
  strncpy(string, OnScan.Name, 12); string[12] = '\0';
  CharCard(KW_OBJECT, 0, string);
  CharCard(KW_ORIGIN, 0, "DRP");
  if (csys == EQUATORIAL) RealCard(KW_EQUINOX, 0, (double)OnScan.Equinox);
  strncpy(string, OnScan.Observer, 16); string[16] = '\0';
  CharCard(KW_OBSERVER, 0, string);
  VoidCard(KW_END);

  writeFITSheader(fits);
  free(kw);

  map = (WORD *)calloc(DimX*DimY, sizeof(WORD));
  if (map == (WORD *)NULL) DRPerror("can't allocate map");
  
  for (n = 0; n < DimX*DimY; n++) map[n] = BLANK;
  printf("map[0][0] = %d\n", *map);

  /* p = (struct point *)OnScan.Channels; */
  for (n = 0; n < pts; n++) {
    i = p[n].mapx; 
    j = p[n].mapy; 
    pixel = ((i-xminpix)+(j-yminpix)*DimX)*magnify;
    map[pixel] = (WORD)((p[n].value-bzero)/bscale);
  }

  if (magnify > 1) {
    for (j = 0; j < DimY; j += magnify) {
      for (i = magnify; i < DimX; i += magnify) {
	pixel = j*DimX+i-magnify;
	if (map[pixel] != BLANK && map[pixel+magnify] != BLANK) {
	  diff = map[pixel+magnify] - map[pixel];
	  for (m = 1; m < magnify; m++) {
	    map[pixel+m] = map[pixel] + (WORD)(m*diff/magnify);
	  }
	}
      }
    }
    for (i = 0; i < DimX; i++) {
      for (j = magnify; j < DimY; j += magnify) {
	pixel = (j-magnify)*DimX+i;
	if (map[pixel] != BLANK && map[pixel+magnify*DimX] != BLANK) {
	  diff = map[pixel+magnify*DimX] - map[pixel];
	  for (m = 1; m < magnify; m++) {
	    map[pixel+m*DimX] = map[pixel] + (WORD)(m*diff/magnify);
	  } 
	}
      }
    }
  }

  if (nsmooth) smooth(map, DimX, DimY);
#ifdef BYTESWAP
  /* The FITS standard expects the binary words with the MSB first */
  /* We have to swap them on e.g. an Intel CPU */
  for (i = 0; i < DimX*DimY; i++) {
    swapbytes((char *)&map[i], sizeof(WORD));
  }
#endif

  writeFITSdata(fits, DimX*DimY, sizeof(WORD), (void *)map);
  free(map);
    
  DRPinfo("map written to file '%s'", scanname);

  exit(0);
}
