#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include "drp.h"
#include "options.h"

#undef DEBUG

#define MAXITER 20
#define DIM 3
#define NLINES 5
#define EPS 0.1
  
int nlines = 1;
int xUnit = VELOCITY;
float xMin = 0.0, xMax = 0.0;
float Amplitude[NLINES] = { 0.0, 0.0, 0.0, 0.0, 0.0};
float Centre[NLINES]    = { 0.0, 0.0, 0.0, 0.0, 0.0};
float Width[NLINES]     = { 0.0, 0.0, 0.0, 0.0, 0.0};
float Area[NLINES]      = { 0.0, 0.0, 0.0, 0.0, 0.0};
int aguess = 0, cguess = 0, wguess = 0;

int CenterCh(SCAN *);
int VChannel(double , SCAN *);
int FChannel(double , SCAN *);

char PROGRAM_NAME[] = "gauss";
char description[] = "fit Gaussian line profile";
char required[] = "";
struct _options options[] = {
{ "-help",		"print out this message" },
{ "-vel v1 v2",		"select velocity range" },
{ "-freq f1 f2",	"select frequency range" },
{ "-chan c1 c2",	"select channel range" },
{ "-lines n",           "select number of profiles to fit (<= 5)" },
{ "-ampl amplitude",    "guess amplitude(s) of gaussian" },
{ "-centre centre",     "guess centre(s) of gaussian" },
{ "-width width",       "guess width(s) of gaussian" },
{NULL, NULL }};

void ParseOpts(int *pargc, char ***pargv)
{
    char *opt, *optarg;
    int n;

    opt = (*pargv)[0] + 1;

    if (!strcmp(opt, "help")) {
	Help();
	exit(0);
    }
    if (!strcmp(opt, "vel")) {
	GetOption(&optarg, pargc, pargv);
	xMin = atof(optarg);
	GetOption(&optarg, pargc, pargv);
	xMax = atof(optarg);
	xUnit = VELOCITY;
	return;
    }
    if (!strcmp(opt, "freq")) {
	GetOption(&optarg, pargc, pargv);
	xMin = atof(optarg);
	GetOption(&optarg, pargc, pargv);
	xMax = atof(optarg);
	xUnit = FREQUENCY;
	return;
    }
    if (!strcmp(opt, "chan")) {
	GetOption(&optarg, pargc, pargv);
	xMin = (double)atoi(optarg);
	GetOption(&optarg, pargc, pargv);
	xMax = (double)atoi(optarg);
	xUnit = CHANNELS;
	return;
    }
    if (!strcmp(opt, "lines")) {
	GetOption(&optarg, pargc, pargv);
	nlines = atoi(optarg);
	return;
    }
    if (!strcmp(opt, "ampl")) {
	for (n = 0; n < nlines; n++) {
	    GetOption(&optarg, pargc, pargv);
	    Amplitude[n] = atof(optarg);
	}
	aguess = 1;
	return;
    }
    if (!strcmp(opt, "centre")) {
	for (n = 0; n < nlines; n++) {
	    GetOption(&optarg, pargc, pargv);
	    Centre[n] = atof(optarg);
	}
	cguess = 1;
	return;
    }
    if (!strcmp(opt, "width")) {
	for (n = 0; n < nlines; n++) {
	    GetOption(&optarg, pargc, pargv);
	    Width[n] = atof(optarg);
	}
	wguess = 1;
	return;
    }

    Syntax(**pargv);
}

SCAN OnScan, Fit;

/* standard deviation = half width/sqrt(8*ln2) */
#define W 1.665109222  /* sqrt (4*ln2) */

double fgauss(double x, double a[], double dyda[], int na)
{
    int i;
    double fac, ex, arg;
    double y;

    y = 0.0;
    for (i = 0; i < na; i += 3) {
	arg = W*(x-a[i+1])/a[i+2];
	ex = exp(-arg*arg);
	fac = 2.0*a[i]*ex*arg;
	y += a[i]*ex;
	dyda[i] = ex;
	dyda[i+1] = W*fac/a[i+2];
	dyda[i+2] = fac*arg/a[i+2];
    }
    return y;
}

void mrqmin(double [], double [], double [], int , double [], int [],
	int , double **, double **, double *,
	double (*)(double, double [], double [], int), double *);

int main(int argc, char *argv[])
{
    FILE *lf;
    float offx, offy;
    double x;
    static double *X, *Y, *sig, max;
    static double a[NLINES*DIM], da[NLINES*DIM], dy[NLINES*DIM];
    static double **covar, **alpha;
    static double chisq, ochisq, alamda;
    static int ia[NLINES*DIM];
    int dof, itst;
    int i, j, iter, cc = 0, ma;
    int FirstCh = 0, LastCh = 0;
    double var, sdev;

    GetOpts(argc, argv);

    GetScan (&OnScan);
    if (xMin == 0.0 && xMax == 0.0) {
	FirstCh = 0;
	LastCh = OnScan.NChannel-1;
    } else {
	switch (xUnit) {
	  case VELOCITY:
	    FirstCh = VChannel((double)xMin, &OnScan);
	    LastCh  = VChannel((double)xMax, &OnScan);
	    break;
	  case FREQUENCY:
	    FirstCh = FChannel((double)xMin, &OnScan);
	    LastCh  = FChannel((double)xMax, &OnScan);
	    break;
	  case CHANNELS:
	    FirstCh = (int)xMin;
	    LastCh  = (int)xMax;
	    break;
	}
    }

    if (FirstCh > LastCh) {
	i = FirstCh;
	FirstCh = LastCh;
	LastCh = i;
    }

    max = OnScan.Channels[0];
    for (i = 1; i < OnScan.NChannel; i++) {
	if (OnScan.Channels[i] > max) {
	    max = OnScan.Channels[i];
	    cc = i;
	}
    }
    if (!aguess)  Amplitude[0] = max;
    if (!cguess)  Centre[0] = (double)cc;
    if (!wguess)  Width[0] = 10.0;

    for (i = 0; i < nlines; i++) {
	switch (xUnit) {
	  case VELOCITY: 
	    if (cguess) Centre[i] = VChannel((double)Centre[i], &OnScan);
	    if (wguess) Width[i] /= fabs(OnScan.VelRes);
	    break;
	  case FREQUENCY: 
	    if (cguess) Centre[i] = FChannel((double)Centre[i], &OnScan);
	    if (wguess) Width[i] /= fabs(OnScan.FreqRes);
	    break;
	  case CHANNELS: 
	    break;
	}
    }
  
    if (FirstCh > LastCh) {
	i = FirstCh;
	FirstCh = LastCh;
	LastCh = i;
    }
 
    ma = DIM*nlines;
    dof = LastCh-FirstCh+1;
    if (dof <= ma) DRPerror("too few points to fit");
  
    DRPinfo("guess values [channels]:");
    for (i = 0; i < nlines; i++) {
	if (nlines > 1) printf(" line #%d\n", i+1);
	printf (" Amplitude = %f\n", Amplitude[i]);
	printf (" Centre    = %f\n", Centre[i]);
	printf (" Width     = %f\n", Width[i] );
    }
    printf (" fit between channels %d and %d\n", FirstCh, LastCh);
  
    covar = (double **)calloc(ma, sizeof(double *));
    if (covar == NULL) DRPerror("memory allocation failure");
    for (i = 0; i < ma; i++) {
        covar[i] = (double *)calloc(ma, sizeof(double));
        if (covar[i] == NULL) DRPerror("memory allocation failure");
    }

    alpha = (double **)calloc(ma, sizeof(double *));
    if (alpha == (double **)NULL) DRPerror("memory allocation failure");
    for (i = 0; i < ma; i++) {
        alpha[i] = (double *)calloc(ma, sizeof(double));
        if (alpha[i] == (double *)NULL) DRPerror("memory allocation failure");
    }

    alamda = -1.0;

    X = (double *)calloc(dof, sizeof(double));
    if (X == (double *)NULL) DRPerror("memory allocation failure");

    Y = (double *)calloc(dof, sizeof(double));
    if (Y == (double *)NULL) DRPerror("memory allocation failure");

    sig = (double *)calloc(dof, sizeof(double));
    if (sig == (double *)NULL) DRPerror("memory allocation failure");

    for (j = 0, i = FirstCh; i <= LastCh; j++, i++) {
	X[j] = (double)i;
	Y[j] = (double)OnScan.Channels[i];
	sig[j] = 1.0;
    }

    for (i = 0; i < nlines; i++) {
	a[3*i+0] = (double)Amplitude[i];    ia[3*i+0] = 1;
	a[3*i+1] = (double)Centre[i];       ia[3*i+1] = 1;
	a[3*i+2] = (double)Width[i];        ia[3*i+2] = 1;
    }

    alamda = -1;
    mrqmin(X, Y, sig, dof, a, ia, ma, covar, alpha, &chisq, fgauss, &alamda);
    iter = 0;
    itst = 0;
    while (itst < 5) {
#ifdef DEBUG
	printf("Iteration #%2d(%2d) chi2 %8.4f(%8.4f) alambda %9.2e\n", 
	       iter, itst, chisq, ochisq, alamda);
	printf("  a[0]     a[1]     a[2]   \n");
	for (i = 0; i < ma; i++) printf("%9.4f", a[i]);
	printf("\n");       
#endif
	iter++;
	ochisq = chisq;
	mrqmin(X,Y,sig,dof,a,ia,ma,covar,alpha,&chisq,fgauss,&alamda);
	if (chisq > ochisq)	            itst = 0;
	else if (fabs(ochisq-chisq) < 0.001)  itst++;
#ifdef DEBUG
	printf("    %f\n", fabs(ochisq-chisq));       
#endif
    }

    alamda = 0.0;
    mrqmin(X, Y, sig, dof, a, ia, ma, covar, alpha, &chisq, fgauss, &alamda);

    for (i = 0; i < ma; i++) 
      da[i] = sqrt(covar[i][i]);
  
    memcpy(&Fit, &OnScan, sizeof(SCAN));
    for (i = 0; i < Fit.NChannel; i++) {
	x = (double)i;
	Fit.Channels[i] = (float)fgauss(x, a, dy, ma);
    }
    for (i = 0; i < ma; i++) Fit.work[i] = a[i];
    PutTemp (&Fit);

    sdev = var = 0.0;
    for (i = FirstCh; i <= LastCh; i++) {
	sdev = OnScan.Channels[i] - Fit.Channels[i];
	var += sdev*sdev;
    }
    var /= (double)(LastCh - FirstCh);
    sdev = sqrt(var);
    for (i = 0; i < ma; i++) da[i] *= sdev;

    DRPinfo("converged after %d iterations", iter);
    printf (" fit results:\n");
    for (i = 0; i < nlines; i++) {
	if (nlines > 1) printf(" line #%d\n", i+1);
	printf (" Amplitude = %f (%f) [K]\n", a[3*i], da[3*i]);
	Area[i] = 1.064467*a[3*i]*a[3*i+2]; /* sqrt(PI/4ln(2)) * Amp * Width */
	if (argc == 1) xUnit = VELOCITY;
	switch (xUnit) {
	  case CHANNELS:
	    printf (" Centre    = %f (%f) [channel]\n", a[3*i+1], da[3*i+1]);
	    printf (" Width     = %f (%f) [channels]\n", a[3*i+2], da[3*i+2]);
	    printf (" Area      = %f [K*channels]\n", Area[i]);
	    break;
	  case VELOCITY:
	    a[3*i+1] = (a[3*i+1]-CenterCh(&OnScan))
	      *OnScan.VelRes+OnScan.VSource;
	    a[3*i+2] *= fabs(OnScan.VelRes);
	    da[3*i+1] *= fabs(OnScan.VelRes);
	    da[3*i+2] *= fabs(OnScan.VelRes);
	    Area[i]   *= fabs(OnScan.VelRes);
	    printf (" Centre    = %f (%f) [km/s]\n", a[3*i+1], da[3*i+1]);
	    printf (" Width     = %f (%f) [km/s]\n", a[3*i+2], da[3*i+2]);
	    printf (" Area      = %f [K*km/s]\n", Area[i]);
	    break;
	  case FREQUENCY:
	    a[3*i+1] = (a[3*i+1]-CenterCh(&OnScan))
	      *OnScan.FreqRes+OnScan.RestFreq;
	    a[3*i+2] *= fabs(OnScan.FreqRes);
	    da[3*i+1] *= fabs(OnScan.FreqRes);
	    da[3*i+2] *= fabs(OnScan.FreqRes);
	    Area[i]   *= fabs(OnScan.FreqRes);
	    printf (" Centre    = %f (%f) [MHz]\n", a[3*i+1], da[3*i+1]);
	    printf (" Width     = %f (%f) [MHz]\n", a[3*i+2], da[3*i+2]);
	    printf (" Area      = %f [K*MHz]\n", Area[i]);
	    break;
	}
    }
    printf(" root mean sqare deviation of fit: %f [K]\n", sdev);
  
    offx = OnScan.LMapOff*RADTOMIN;
    offy = OnScan.BMapOff*RADTOMIN;
    if ((lf = fopen("drp.log","a")) != NULL) {
	for (i = 0; i < nlines; i++) {
	    fprintf (lf, "GAUSS:\t%04d", OnScan.ScanNo);
	    fprintf (lf, "\t%10.3f\t%10.3f", offx, offy);
	    fprintf (lf,"\t%10.3f\t%10.3f\t%10.3f\t%10.3f\n",
		     a[3*i+0], a[3*i+1], a[3*i+2], Area[i]);
	}
    } else DRPerror("could not append to log file");
  
    free(sig);
    free(Y);
    free(X);
    for (i = 0; i < ma; i++) free(alpha[i]);
    free(alpha);
    for (i = 0; i < ma; i++) free(covar[i]);
    free(covar);

    exit (0);
}
