#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "drp.h"
#include "options.h"
#include "dfil.h"

static DFWORD scanno = 0;
static DFWORD block = 0;
#ifdef CONVEX
static int dfil = 0;
#endif
static int receiver=0;
static int blockno=0;
static int blocklen = 3;
char dfilname[48];

char PROGRAM_NAME[] = "getants";
char description[] = "get scan from ANTS file (dfile)";
char required[] = "";
struct _options options[] = {
{ "-help",			"Print out this message" },
#ifdef CONVEX
{ "-dfile",                      "specify 'dfile' number" },
#endif
{ "-file",                      "specify file name" },
{ "-scan n",                    "specify scan number" }, 
{ "-block n",                   "specify block number" }, 
{ "-rec x",                     "specify receiver (x = A or B)" },
{ "-size n",                    "specify blocksize" }, 
{NULL, NULL }};

void ParseOpts(int *pargc, char ***pargv)
{
    char *opt, *optarg;

    opt = (*pargv)[0] + 1;

    if (!strcmp(opt, "help")) {
	Help();
	exit(0);
    }
#ifdef CONVEX
    if (!strcmp(opt, "dfile")) {
	GetOption(&optarg, pargc, pargv);
	dfil = atoi(optarg);
	return;
    }
#endif
    if (!strcmp(opt, "file")) {
	GetOption(&optarg, pargc, pargv);
	strcpy(dfilname, optarg);
	return;
    }
    if (!strcmp(opt, "scan")) {
	GetOption(&optarg, pargc, pargv);
	scanno = atoi(optarg);
	blockno = 0;
	return;
    }
    if (!strcmp(opt, "block")) {
	GetOption(&optarg, pargc, pargv);
	block = atoi(optarg);
	blockno = 1;
	return;
    }
    if (!strcmp(opt, "size")) {
	GetOption(&optarg, pargc, pargv);
	blocklen = atoi(optarg);
	return;
    }
    if (!strcmp(opt, "rec")) {
	GetOption(&optarg, pargc, pargv);
	if (toupper(optarg[0]) == 'B') receiver = 1;
	else                           receiver = 0;
	return;
    }
    Syntax(**pargv);
}

DFWORD indx[DFILBLOCK];

SCAN OnScan;

void hms(float rad, short *hour, short *min, short *sec)
{
    long int seconds;

    seconds = (long int)(rad*RADTOSEC/15.0);
    *hour = (DFWORD) (seconds/3600);
    *min = (DFWORD) ((seconds/60)%60);
    *sec = (DFWORD) (seconds%60);
}

#define HEADERLEN 150

DFWORD *itwh;
float *twh;

short int dfint(int n)
{
    static DFWORD value;

    memcpy(&value, itwh+n, sizeof (DFWORD));
/*  printf("short  value[%3d]: %d\n", n, value); */
    return (value);
}

float dffloat(int n)
{
    static float value;

    memcpy(&value, itwh+n, sizeof (float));
#ifndef CONVEX
    value /= 4.0;
#endif
/*  printf("float  value[%3d]: %f\n", n, value); */
    return (value);
}

double dfdble(int n)
{
    static double value;

    memcpy(&value, itwh+n, sizeof (double));
#ifndef CONVEX
    value /= 4.0;
#endif
/*  printf("double value[%3d]: %lf\n", n, value); */
    return (value);
}

void filltwh (SCAN *scan, int rec)
{
    int i;
    float *next;

    twh = (float *)itwh;
    scan->ScanNo = dfint(0);
    scan->NChannel = (int)dffloat(2*(55+rec));
    scan->Day = dfint(28);
    scan->Month = dfint(27);
    scan->Year = dfint(29); if (scan->Year < 1900) scan->Year += 1900;
    hms(dffloat(2*12), &scan->UTHour, &scan->UTMin, &scan->UTSec);
    hms(dffloat(2*11), &scan->STHour, &scan->STMin, &scan->STSec);
    scan->ObsMode = dfint(30);
    scan->CSystem = 0;
    strncpy (scan->Name, (char *)(itwh+12), 12);
    strncpy (scan->Project, (char *)(itwh+8), 4);
    strncpy (scan->Molecule, (char *)(itwh+260), 18);
/*  scan->JulDate = (int)dffloat(2*9); */
    scan->Longitude = dfdble(42);
    scan->Latitude = dfdble(45);
    scan->LMapOff = dffloat(2*90);
    scan->BMapOff = dffloat(2*91);
    scan->AzMapOff = dffloat(2*88);
    scan->ElMapOff = dffloat(2*89);
    scan->Azimuth = dfdble(164);
    scan->Elevation = dfdble(167);
    scan->Equinox = dffloat(2*24);
    scan->Tsys = dffloat(2*(45+rec));
    scan->Tcal = dffloat(2*(41+rec));
    scan->Tau = dffloat(2*129);
    scan->IntTime = dffloat(2*(47+rec));
    scan->AirTemp = dffloat(2*77);
    scan->Pressure = dffloat(2*78);
    scan->Humidity = dffloat(2*79);
    scan->VSource = (double) dffloat(2*(49+rec));
    scan->VelRes = (double) dffloat(2*(51+rec));
    scan->RestFreq = dfdble(70+3*rec);
    scan->SkyFreq = dfdble(76+3*rec);
    scan->Bandwidth = (double) dffloat(2*(53+rec));
    scan->FreqRes = scan->Bandwidth/scan->NChannel;
    next = twh+150; 
    if (rec) next += (int)dffloat(2*55);
    for (i=0; i<scan->NChannel; i++) {
	scan->Channels[i] = *next++;
#ifndef CONVEX
	scan->Channels[i] /= 4.0;
#endif
    }
}

int main(int argc, char *argv[])
{
    FILE *df = NULL;

    GetOpts(argc, argv);

    if (dfilname[0] == '\0') {
#ifdef CONVEX
	if (dfil == 0 || (scanno == 0 && block == 0)) 
	  DRPerror("file and scan (or block) number required");

	who = getenv("USER");
	df = OpenDfil(dfil, blocklen, who, dfilname);
#endif
    } else df = fopen(dfilname, "r+");
    if (df == NULL) DRPerror("can't open dfile %s", dfilname);

    ReadDfilIndex(df, indx);
    if (!blockno) block = FindDfilScan (scanno,indx);
    else        { scanno = indx[block-1]; }
    if (block == 0) DRPerror("scan %d not found", scanno);

    itwh = (DFWORD *)calloc((unsigned)(blocklen*DFILBLOCK),sizeof(DFWORD));
    if (itwh == NULL) DRPerror("can't allocate scan");

    GetDfilBlock (df, block, blocklen, itwh);
    filltwh(&OnScan, receiver); 
    OnScan.Version = DRPVERSION; /* indicate DRP version */

    PutScan (&OnScan);
    DRPinfo("scan %d found in dfil '%s'", scanno, dfilname);
    exit (0);
}
