#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <ctype.h>
#include "drp.h"
#include "options.h"

#ifdef sun
#define memmove memcpy
#endif

static char drspname[MAXNAMLEN+1] = "";
int scan = 0;
int record = 0;

char PROGRAM_NAME[] = "getdrsp";
char description[] = "extract scan from a DRAWSPEC data file";
char required[] = "";
struct _options options[] = {
{ "-help",			"Print out this message" },
{ "-file filename",             "specify DRAWSPEC file name" },
{ "-scan n",                    "specify scan number" }, 
{ "-record n",                  "specify record number" }, 
{NULL, NULL }};

void ParseOpts(int *pargc, char ***pargv)
{
    char *opt, *optarg;

    opt = (*pargv)[0] + 1;

    if (!strcmp(opt, "help")) {
	Help();
	exit(0);
    }
    if (!strcmp(opt, "file")) {
	GetOption(&optarg, pargc, pargv);
	strcpy(drspname, optarg);
	return;
    }
    if (!strcmp(opt, "scan")) {
	GetOption(&optarg, pargc, pargv);
	scan = atoi(optarg);
	return;
    }
    if (!strcmp(opt, "record")) {
	GetOption(&optarg, pargc, pargv);
	record = atoi(optarg);
	return;
    }
    Syntax(**pargv);
}

SCAN OnScan;

#define MAXBLOCK 1000
#define MAXCHAN  1024

void revb(char *from, int nbytes)
{
    register char *p, *q, c;
    register int i;

    for (i=0, p=from, q=p+nbytes; i<nbytes/2; i++) {
	q--;
	c = *p;
	*p = *q;
	*q = c;
	p++;
    }
}

short int pcint(short int x) 
{
    revb ((char *)&x, sizeof(short int));
    return x;
}

double pcdble(double x)
{
    revb((char *)&x, sizeof(double));
    return x;
}

#define DRSPHEADER 191

struct MyFormat {
    char   Source[13], OrigSource[13], date1[9], date2[9], lst[9];
    short  Scan_number, OrigScan_number, Nchan, TheCode, TheOldCode;
    double TC, TS, Int_time, RmsNow;
    double Az, El, Ra, Decl, Gl, Gb, BSum, BDif;
    double Rest_freq, Delta_freq, Center_v, Delta_v;
    /* DRSPHEADER bytes up to here */
    short  spectra[MAXCHAN];
    long   next;
} drsp;

/*
 * unfortunately the 'MyFormat' has all record members after drsp.lst
 * aligned on odd byte boundaries. We have to shift all those members
 * by one byte, assuming that the C compiler will have introduced a 
 * pad byte.
 */ 
void align(void)
{
    char *p;
    int len;

    p = drsp.lst+9;
    len = sizeof(struct MyFormat)-(p-drsp.Source);
    memmove(p+1, p, len);
}

void pas2cstr(char *to, char *from)
{
    int len;

    len = (int)from[0];
    strncpy(to, from+1, len);
    to[len] = '\0';
}

static double bzero, bscale;

int filldrsp(void)
{
    int n;
    char date[9];

    align();
    pas2cstr(OnScan.Name, drsp.Source);
    pas2cstr(date, drsp.date2);
    sscanf(date, "%2hd/%2hd/%2hd", &OnScan.Month, &OnScan.Day, &OnScan.Year);
    OnScan.Year += 1900;
    pas2cstr(date, drsp.lst);
    sscanf(date, "%2hd:%2hd:%2hd",&OnScan.STHour,&OnScan.STMin,&OnScan.STSec);

    OnScan.ScanNo = pcint(drsp.Scan_number);
    OnScan.NChannel = pcint(drsp.Nchan)+1;

    OnScan.Tcal = (float)pcdble(drsp.TC);
    OnScan.Tsys = (float)pcdble(drsp.TS);
    OnScan.IntTime = (float)pcdble(drsp.Int_time);

    OnScan.Azimuth = (float)pcdble(drsp.Az)/RADTODEG;
    OnScan.Elevation = (float)pcdble(drsp.El)/RADTODEG;
    OnScan.Longitude = (float)pcdble(drsp.Ra)*15.0/RADTODEG;
    OnScan.Latitude = (float)pcdble(drsp.Decl)/RADTODEG;
    OnScan.GalLong = (float)pcdble(drsp.Gl)/RADTODEG;
    OnScan.GalLat = (float)pcdble(drsp.Gb)/RADTODEG;
    drsp.RmsNow = pcdble((double)OnScan.Tsys/sqrt(OnScan.Bandwidth*OnScan.IntTime*1.0e6));
    bzero = pcdble(drsp.BSum);
    bscale = pcdble(drsp.BDif);

    OnScan.RestFreq = pcdble(drsp.Rest_freq);
    OnScan.FreqRes = pcdble(drsp.Delta_freq);
    OnScan.VSource = pcdble(drsp.Center_v);
    OnScan.VelRes = pcdble(drsp.Delta_v);
    OnScan.Bandwidth = OnScan.FreqRes*OnScan.NChannel;

    /* header information which can't be recovered from DRAWSPEC */
    OnScan.ObsMode = 0;
    OnScan.CSystem = 0;
    OnScan.Equinox = 1950.0;
    OnScan.EquiNow = 0.0;
    OnScan.AirTemp = 0.0;
    OnScan.Pressure = 0.0;
    OnScan.Humidity = 0.0;
    OnScan.Long2000 = 0.0;
    OnScan.Lat2000 = 0.0;
    OnScan.LMapOff = 0.0;
    OnScan.BMapOff = 0.0;

    n = pcint(drsp.TheCode)+1;
    return (n);
}

int main(int argc, char *argv[])
{
    FILE *drspfile;
    int i, j, n, len;

    GetOpts(argc, argv);
    if (drspname[0] == '\0') {
	if (argc <= 1) {
	    Syntax("");
	    exit(1);
	} else {
	    strcpy(drspname, argv[1]);
	}
    }

    drspfile = fopen(drspname, "r");
    if (drspfile == NULL) DRPerror("can't append to '%s'", drspname);
    
    if (scan == 0 && record == 0) record = 1;
    if (scan != 0) record = MAXBLOCK;
    for (i = 0; i < record; i++) {
	len = fread((void *)&drsp, sizeof(char), DRSPHEADER, drspfile);
	if (len != DRSPHEADER) DRPerror("read error (%d)", len);
	n = filldrsp();
	len = fread((void *)drsp.spectra, sizeof(short), n, drspfile);
	if (len != n) DRPerror("read error (%d)", len);
	len = fread((void *)&drsp.next, sizeof(long), 1, drspfile);
	if (len != 1) DRPerror("read error (%d)", len);
    
	for (j = 0; j < OnScan.NChannel; j++) {
	    OnScan.Channels[j] = (float)(bscale*pcint(drsp.spectra[j])+bzero);
	}
	if (OnScan.ScanNo == scan) break;
    }
    PutScan(&OnScan);

    DRPinfo("scan extracted from file '%s'", drspname);

    exit(0);
}
