#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include "drp.h"
#include "options.h"
#include "fits.h"

static char scanname[MAXNAMLEN+1] = "";

char PROGRAM_NAME[] = "getfits";
char description[] = "get scan in FITS format";
char required[] = "[scanname]";
struct _options options[] = {
{ "-help",			"Print out this message" },
{ "-file filename",             "specify full filename" },
{NULL, NULL }};

void ParseOpts(int *pargc, char ***pargv)
{
  char *opt, *optarg;

  opt = (*pargv)[0] + 1;

  if (!strcmp(opt, "help")) {
    Help();
    exit(0);
  }

  if (!strcmp(opt, "file")) {
    GetOption(&optarg, pargc, pargv);
    strcpy(scanname, optarg);
    return;
  }
  Syntax(**pargv);
}

typedef short int WORD;
typedef long int LONG;

static SCAN OnScan;

extern int used;
struct fitskey *readFITSheader(FILE *);
int readFITSdata(FILE *, int ,int ,void *);
#ifdef BYTESWAP
void swapbytes(char *, int );
#endif

#define VLSR (0<<2)
#define VHEL (1<<2)
#define VGEO (2<<2)

int main(int argc, char *argv[])
{
  FILE *fits;
  void *data = NULL;
  short *ip;
  long *jp;
  float bzero = 0.0, bscale = 0.0;
  double rpix = 0.0;
  int i, n, no, size;
  static int /* simple, extend, */ bitpix, naxis;
  char *path, *ptr;
  struct fitskey *kw;

  GetOpts(argc, argv);

  if (scanname[0] == '\0') {
    if (argv[1] == NULL) {
      Syntax("");
      exit(EX_ARGSBAD);
    } else {
      if ((path = getenv("FITSAREA")) == NULL) { 
	strcpy(scanname, argv[1]);
	strcat(scanname, ".fits");
      } else {
	strcpy(scanname, path);
	if (path[strlen(path)-1] != '/') strcat(scanname, "/");
	strcat(scanname, argv[1]);
	strcat(scanname, ".fits");
      }
    }
  }

    
  fits = fopen(scanname, "r");             /* open the file for reading  */
  if (fits == NULL) DRPerror("can't open file '%s'", scanname);

  used = 0;
  kw = readFITSheader(fits);
  if (kw == NULL) DRPerror("can't read fits header");

  for (i = 0; i < used; i++) {
#ifdef DEBUG
    printf("known keyword: %s [%d]\n", known[kw[i].key].text, kw[i].dim);
#endif
    switch (kw[i].key) {
     case KW_SIMPLE:
      /* simple = kw[i].val.l; */
      break;
     case KW_EXTEND:
      /* extend = kw[i].val.l; */
      break;
     case KW_BITPIX:
      bitpix = kw[i].val.l;
      break;
     case KW_NAXIS:
      if (kw[i].dim == 0) naxis = kw[i].val.l;
      else {
	if (kw[i].dim  > naxis) {
	  DRPwarning("invalid NAXIS%d keyword", kw[i].dim);
	} else {
	  if (kw[i].dim == 1) {
	    OnScan.NChannel = kw[i].val.l;
	    OnScan.Ctrl = 0;
	  } else if (kw[i].val.l > 1) DRPwarning("not a spectrum");
	}
      }
      break;
     case KW_CTYPE:
      switch (kw[i].dim) {
       case 1:
	if (strncmp(kw[i].val.str, "FREQ", 4)) 
	  DRPerror("frequency required on 1st axis");
	break;
       case 2:
	if (strncmp(kw[i].val.str, "RA", 2)) {
	  if (strncmp(kw[i].val.str, "GLON", 4)) {
	    DRPerror("RA or GLON required on 2nd axis");
	  } else  OnScan.CSystem = 5;
	} else OnScan.CSystem = 2;
	break;
       case 3:
	if (strncmp(kw[i].val.str, "DEC", 3)) {
	  if (strncmp(kw[i].val.str, "GLAT", 4)) {
	    DRPerror("DEC or GLAT required on 3rd axis");
	  } else OnScan.CSystem = 5;
	} else OnScan.CSystem = 2;
	break;
      }
      break;
     case KW_CRPIX:
      switch (kw[i].dim) {
       case 1:
	rpix = kw[i].val.d;
	break;
      }
      break;
     case KW_CRVAL:
      switch (kw[i].dim) {
       case 1:
	OnScan.RestFreq += kw[i].val.d/1.0e6;    /* Hz -> MHz */
	break;
       case 2:
	OnScan.Longitude = kw[i].val.d*PI/180.0;
	break;
       case 3:
	OnScan.Latitude = kw[i].val.d*PI/180.0;
	break;
      }
      break;
     case KW_CDELT:
      switch (kw[i].dim) {
       case 1:
	OnScan.FreqRes   = kw[i].val.d/1.0e6;
	break;
       case 2:
	OnScan.LMapOff   = kw[i].val.d*PI/180.0;
	break;
       case 3:
	OnScan.BMapOff  = kw[i].val.d*PI/180.0;
	break;
      }
      break;
     case KW_CROTA:
      break;
     case KW_BSCALE:
      bscale = kw[i].val.d;
      break;
     case KW_BZERO:
      bzero  = kw[i].val.d;
      break;
     case KW_BUNIT:
      break;
     case KW_BLANK:
      break;
     case KW_DATAMAX:
      break;
     case KW_DATAMIN:
      break;
     case KW_ORIGIN:
      break;
     case KW_OBSERVER:
      strncpy(OnScan.Observer ,kw[i].val.str, 16);
      break;
     case KW_OBJECT:
      strncpy(OnScan.Name, kw[i].val.str, 12);
      break;
     case KW_INSTRUME:
      break;
     case KW_TELESCOP:
      break;
     case KW_LINE:
      strncpy(OnScan.Molecule ,kw[i].val.str, 16);
      break;
     case KW_SCAN:
      no = (int)(kw[i].val.d) % 10000;
      OnScan.ScanNo = (short)no;
      break;
     case KW_UTC:
      ptr = strtok(kw[i].val.str,":\0");
      OnScan.UTHour = atoi(ptr);
      ptr = strtok(NULL,":\0");
      OnScan.UTMin = atoi(ptr);
      ptr = strtok(NULL,":\0");
      OnScan.UTSec = atoi(ptr);
      /*	
		kw[i].val.d = 60.0*modf(kw[i].val.d, &ip);
		OnScan.UTHour = (short)ip;
		kw[i].val.d = 60.0*modf(kw[i].val.d, &ip);
		OnScan.UTMin  = (short)ip;
		kw[i].val.d = modf(kw[i].val.d, &ip);
		OnScan.UTSec  = (short)ip; 
		*/
      break;
     case KW_LST:
      ptr = strtok(kw[i].val.str,":\0");
      OnScan.STHour = atoi(ptr);
      ptr = strtok(NULL,":\0");
      OnScan.STMin = atoi(ptr);
      ptr = strtok(NULL,":\0");
      OnScan.STSec = atoi(ptr);
      break;
     case KW_DATE:
      break;
     case KW_DATE_OBS:
      /* check for the newly adopted date representation in FITS */
      /* expected format CCYY-MM-DD or CCYY-MM-DDThh:mm:ss.sss   */
      if (strchr(kw[i].val.str,'-')) {
	ptr = strtok(kw[i].val.str,"-");
	OnScan.Year = atoi(ptr);
	ptr = strtok(NULL,"-");
	OnScan.Month = atoi(ptr);
	ptr = strtok(NULL,"T");
	OnScan.Day = atoi(ptr);
	ptr = strtok(NULL,":");
	if (ptr) {
	  OnScan.UTHour = atoi(ptr);
	  ptr = strtok(NULL,":");
	  OnScan.UTMin = atoi(ptr);
	  ptr = strtok(NULL,":");
	  OnScan.UTSec = atoi(ptr);
	}
      }
      /* check for old representation DD/MM/YY */
      else if (strchr(kw[i].val.str,'/')) {
	ptr = strtok(kw[i].val.str,"/");
	OnScan.Day = atoi(ptr);
	ptr = strtok(NULL,"/");
	OnScan.Month = atoi(ptr);
	ptr = strtok(NULL,"/");
	OnScan.Year = atoi(ptr);
	if (OnScan.Year < 100) OnScan.Year += 1900;
      } 
      break;
     case KW_JDATE:
      OnScan.JulDate = (long)(kw[i].val.d+0.5);
      break;
     case KW_TSYS:
      OnScan.Tsys = kw[i].val.d;
      break;
     case KW_TLOAD:
      OnScan.Tcal = kw[i].val.d;
      break;
     case KW_TREC:
      OnScan.Trec = kw[i].val.d;
      break;
     case KW_TAU:
      OnScan.Tau  = kw[i].val.d;
      break;
     case KW_DBLOAD:
      OnScan.dBl  = kw[i].val.d;
      break;
     case KW_EQUINOX:
      OnScan.Equinox = kw[i].val.d;
      break;
     case KW_RA:
      OnScan.Long2000 = kw[i].val.d*PI/180.0;
      break;
     case KW_DEC:
      OnScan.Lat2000 = kw[i].val.d*PI/180.0;
      break;
     case KW_AZIMUTH:
      OnScan.Azimuth = kw[i].val.d*PI/180.0;
      break;
     case KW_ELEVATIO:
      OnScan.Elevation = kw[i].val.d*PI/180.0;
      break;
     case KW_AZOFF:
      OnScan.AzMapOff = kw[i].val.d*PI/180.0;
      break;
     case KW_ELOFF:
      OnScan.ElMapOff = kw[i].val.d*PI/180.0;
      break;
     case KW_AZPOINT:
      OnScan.AzPointg = kw[i].val.d*PI/180.0;
      break;
     case KW_ELPOINT:
      OnScan.ElPointg = kw[i].val.d*PI/180.0;
      break;
     case KW_AZCORR:
      OnScan.AzOffset = kw[i].val.d*PI/180.0;
      break;
     case KW_ELCORR:
      OnScan.ElOffset = kw[i].val.d*PI/180.0;
      break;
     case KW_COLLIMAT:
      OnScan.AzOffset += kw[i].val.d*PI/180.0
	/cos(OnScan.Elevation);
      break;
     case KW_RESTFREQ:
      OnScan.RestFreq += kw[i].val.d/1.0e6;  /* Hz -> MHz */
      break;
     case KW_VLSR:
     case KW_VELO_LSR:
      OnScan.VSource = kw[i].val.d/1000.0;   /* m/s -> km/s */
      OnScan.Ctrl |= VLSR;
      break;
     case KW_VHEL:
     case KW_VELO_HEL:
      OnScan.VSource = kw[i].val.d/1000.0;   /* m/s -> km/s */
      OnScan.Ctrl |= VHEL;
      break;
     case KW_VELO_GEO:
      OnScan.VSource = kw[i].val.d/1000.0;   /* m/s -> km/s */
      OnScan.Ctrl |= VGEO;
      break;
     case KW_DELTAV:
      OnScan.VelRes  = kw[i].val.d/1000.0;  /* m/s -> km/s */
      /* kludge for taking care of inconsistencies in *.fits */
      /* we want df*dv < 0 !!! */
      if (OnScan.VelRes*OnScan.FreqRes > 0.0) 
	OnScan.FreqRes = -OnScan.FreqRes;
      break;
     case KW_OBSTIME:
      OnScan.IntTime = kw[i].val.d;
      break;
     case KW_TAMB:
     case KW_TOUTSIDE:
      OnScan.AirTemp  = kw[i].val.d;
      break;
     case KW_PRESSURE:
      OnScan.Pressure = kw[i].val.d/100.0;  /* Pa -> hPa */
      break;
     case KW_HUMIDITY:
      OnScan.Humidity = kw[i].val.d*100.0;  /*    -> % */
      if (OnScan.Humidity > 100.0) OnScan.Humidity /= 100.0;
      break;
     case KW_POSLONG:
      OnScan.MapX = kw[i].val.l;
      break;
     case KW_POSLAT:
      OnScan.MapY = kw[i].val.l;
      break;
     case KW_SPACELON:
      OnScan.StepX = kw[i].val.d*PI/180.0;
      break;
     case KW_SPACELAT:
      OnScan.StepY = kw[i].val.d*PI/180.0;
      break;
     case KW_MAPTILT:
      OnScan.PosAngle = kw[i].val.d*PI/180.0;
      break;

    }
  }

  OnScan.Bandwidth = fabs(OnScan.FreqRes)*OnScan.NChannel;
  rpix -= (double)CenterCh(&OnScan) + 1.0;
#ifdef DEBUG
  printf("rpix = %lf\n", rpix);
#endif
  OnScan.RestFreq += OnScan.FreqRes*rpix; 
    
  OnScan.SubScan = 0;
  OnScan.ObsMode = 0;
  OnScan.Backend = 0;
  OnScan.Frontend = 0;
    
  OnScan.EquiNow = 0.0;
  OnScan.Long2000 = 0.0;
  OnScan.Lat2000 = 0.0;
    
  OnScan.AzErrAve = 0.0;
  OnScan.ElErrAve = 0.0;
  OnScan.AzErrRms = 0.0;
  OnScan.ElErrRms = 0.0;
  OnScan.GalLong = 0.0;
  OnScan.GalLat = 0.0;
  OnScan.VHel = 0.0;
  OnScan.VLsr = 0.0;
    
  OnScan.RefCorr = 0.0;
  OnScan.ParAngle = 0.0;
    
  OnScan.SkyFreq   = OnScan.RestFreq;
    
  if (OnScan.VelRes == 0.0) 
    OnScan.VelRes  = -C*OnScan.FreqRes/OnScan.RestFreq/1000.0;

  for (i = 0; i < 10; i++) OnScan.work[i] = 0.0;
  for (i = 0; i < 31; i++) OnScan.flag[i] = 0;
    
  free(kw);

  switch (bitpix) {
   case 16:
    n = OnScan.NChannel;
    size = sizeof(WORD);
    data = calloc(n, size);
    ip = (WORD *)data;
    if (data == NULL) DRPerror("memory allocation failure");
    else {
      i = readFITSdata(fits, n, size, data);
      if (i == 0) DRPerror("error reading fits data");
    }
    for (i = 0; i < OnScan.NChannel; i++) {
#ifdef BYTESWAP
      /* The FITS standard expects the binary words with the MSB first */
      /* We have to swap them on e.g. an Intel CPU */
      swapbytes((char *)&ip[i], sizeof(WORD));
#endif
      OnScan.Channels[i] = bzero+bscale*ip[i];
    }
    break;
   case 32:
    n = OnScan.NChannel;
    size = sizeof(LONG);
    data = calloc(n, size);
    jp = (LONG *)data;
    if (data == NULL) DRPerror("memory allocation failure");
    else {
      i = readFITSdata(fits, n, size, data);
      if (i == 0) DRPerror("error reading fits data");
    }
    for (i = 0; i < OnScan.NChannel; i++) {
#ifdef BYTESWAP
      /* The FITS standard expects the binary words with the MSB first */
      /* We have to swap them on e.g. an Intel CPU */
      swapbytes((char *)&jp[i], sizeof(LONG));
#endif
      OnScan.Channels[i] = bzero+bscale*jp[i];
    }
    break;
   default:
    DRPerror("can't read BITPIX = %d format", bitpix);
    break;
  }
  OnScan.Version = DRPVERSION;  /* indicate DRP version */
  PutScan(&OnScan); 
  free(data);
  DRPinfo("scan read from file '%s'", scanname);

  exit(0);
}
