#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include "drp.h"
#include "options.h"
#include "fits.h"

static char scanname[MAXNAMLEN+1] = "";

char PROGRAM_NAME[] = "getfitsmap";
char description[] = "dump FITS map";
char required[] = "[mapname]";
struct _options options[] = {
{ "-help",			"Print out this message" },
{ "-file filename",             "specify full filename" },
{NULL, NULL }};

void ParseOpts(int *pargc, char ***pargv)
{
  char *opt, *optarg;

  opt = (*pargv)[0] + 1;

  if (!strcmp(opt, "help")) {
    Help();
    exit(0);
  }

  if (!strcmp(opt, "file")) {
    GetOption(&optarg, pargc, pargv);
    strcpy(scanname, optarg);
    return;
  }
  Syntax(**pargv);
}

#ifdef BYTESWAP
#define DEFAULTTYPE 0
#else
#define DEFAULTTYPE 1000
#endif

#define STRING (DEFAULTTYPE + 1)
#define NUMBER (DEFAULTTYPE)

/* define names and arrays for MATLAB variables */
static char lon_name[] = "lon";
static char lat_name[] = "lat";
static char map_name[] = "map";

/*
 * savemat - C language routine to save a matrix in a MAT-file.
 *
 * Here is an example that uses 'savemat' to save two matrices to disk,
 * the second of which is complex:
 *
 *	FILE *fp;
 *	double xyz[1000], ar[1000], ai[1000];
 *	fp = fopen("foo.mat","wb");
 *	savemat(fp, 2000, "xyz", 2, 3, 0, xyz, (double *)0);
 *	savemat(fp, 2000, "a", 5, 5, 1, ar, ai);
 *      fclose(fp);
 *
 * Author J.N. Little 11-3-86
 */

typedef struct {
    long type;   /* type */
    long mrows;  /* row dimension */
    long ncols;  /* column dimension */
    long imagf;  /* flag indicating imag part */
    long namlen; /* name length (including NULL) */
} Fmatrix;

savemat(FILE *fp, int type, char *pname, int mrows, int ncols, 
	int imagf, double *preal, double *pimag)
/* FILE *fp;       File pointer */
/* int type;       Type flag: Normally 0 for PC, 1000 for Sun, Mac, and	 */
/*		   Apollo, 2000 for VAX D-float, 3000 for VAX G-float    */
/*		   Add 1 for text variables.	 */
/*		   See LOAD in reference section of guide for more info. */
/* int mrows;      row dimension */ 
/* int ncols;      column dimension */
/* int imagf;	   imaginary flag (0: real array, 1: complex array) */
/* char *pname;    pointer to matrix name */
/* double *preal;  pointer to real data */
/* double *pimag;  pointer to imag data */
{
    int mn;
    Fmatrix x;
	
    x.type = type;
    x.mrows = mrows;
    x.ncols = ncols;
    x.imagf = imagf;
    x.namlen = strlen(pname) + 1;
    mn = x.mrows * x.ncols;

    fwrite(&x, sizeof(Fmatrix), 1, fp);
    fwrite(pname, sizeof(char), (int)x.namlen, fp);
    fwrite(preal, sizeof(double), mn, fp);
    if (imagf) {
	fwrite(pimag, sizeof(double), mn, fp);
    }
}

typedef short int WORD;
typedef long int LONG;

extern int used;
struct fitskey *readFITSheader(FILE *);
int readFITSdata(FILE *, int ,int ,void *);
#ifdef BYTESWAP
void swapbytes(char *, int );
#endif

int main(int argc, char *argv[])
{
  FILE *fits, *fp;
  void *data;
  double *map, *lon, *lat;
  short *ip;
  long *jp;
  float *rp;
  double *dp;
  float bzero, bscale, blank;
  double x, y, dx, dy, bpix, lpix;
  int i, j, k1, k2, n, no, size, dim1, dim2;
  static int simple, extend, bitpix, naxis, xaxis;
  char *path, *ptr;
  struct fitskey *kw;

  GetOpts(argc, argv);

  if (scanname[0] == '\0') {
    if (argv[1] == NULL) {
      Syntax("");
      exit(EX_ARGSBAD);
    } else {
      if ((path = getenv("FITSAREA")) == NULL) { 
	strcpy(scanname, argv[1]);
	strcat(scanname, ".fits");
      } else {
	strcpy(scanname, path);
	if (path[strlen(path)-1] != '/') strcat(scanname, "/");
	strcat(scanname, argv[1]);
	strcat(scanname, ".fits");
      }
    }
  }

  fits = fopen(scanname, "r");             /* open the file for reading  */
  if (fits == NULL) DRPerror("can't open file '%s'", scanname);

  used = 0;
  kw = readFITSheader(fits);
  if (kw == NULL) DRPerror("can't read fits header");

  for (i = 0; i < used; i++) {
#ifdef DEBUG
    printf("known keyword: %s [%d]\n", known[kw[i].key].text, kw[i].dim);
#endif
    switch (kw[i].key) {
     case KW_SIMPLE:
      simple = kw[i].val.l;
      break;
     case KW_EXTEND:
      extend = kw[i].val.l;
      break;
     case KW_BITPIX:
      bitpix = kw[i].val.l;
      break;
     case KW_NAXIS:
      if (kw[i].dim == 0) naxis = kw[i].val.l;
      else {
	if (kw[i].dim  > naxis) {
	  DRPwarning("invalid NAXIS%d keyword", kw[i].dim);
	} else {
	  if (kw[i].dim == 1) dim1 = kw[i].val.l;
	  if (kw[i].dim == 2) dim2 = kw[i].val.l;
	}
      }
      break;
      /*
     case KW_CTYPE:
      switch (kw[i].dim) {
       case 1:
	if (!strncmp(kw[i].val.str, "RA", 2)) break;
	if (!strncmp(kw[i].val.str, "GLON", 2)) break;
	DRPerror("RA or GLAT required on 1st axis");
	break;
       case 2:
	if (!strncmp(kw[i].val.str, "DEC", 2)) break;
	if (!strncmp(kw[i].val.str, "GLAT", 2)) break;
	DRPerror("RA or GLAT required on 1st axis");
	break;
      }
      break;
      */
     case KW_CRPIX:
      switch (kw[i].dim) {
       case 1:
	lpix = kw[i].val.d;
	break;
       case 2:
	bpix = kw[i].val.d;
	break;
      }
      break;
     case KW_CRVAL:
      switch (kw[i].dim) {
       case 1:
	x = kw[i].val.d;    /* degrees */
	break;
       case 2:
	y = kw[i].val.d;    /* degrees */
	break;
      }
      break;
     case KW_CDELT:
      switch (kw[i].dim) {
       case 1:
	dx = kw[i].val.d;    /* degrees */
	break;
       case 2:
	dy = kw[i].val.d;    /* degrees */
	break;
      }
      break;
     case KW_CROTA:
      break;
     case KW_BSCALE:
      bscale = kw[i].val.d;
      break;
     case KW_BZERO:
      bzero  = kw[i].val.d;
      break;
     case KW_BUNIT:
      break;
     case KW_BLANK:
      blank = kw[i].val.d;
      break;
     case KW_DATAMAX:
      break;
     case KW_DATAMIN:
      break;
    }
  }

#ifdef DEBUG
  printf("lpix = %lf\n", lpix);
  printf("bpix = %lf\n", bpix);
#endif

  free(kw);
  lon = calloc(dim1, sizeof(double));
  if (lon == NULL) DRPerror("memory allocation failure");
  lat = calloc(dim2, sizeof(double));
  if (lat == NULL) DRPerror("memory allocation failure");
  for (i = 0; i < dim1; i++) lon[i] = x + dx*(i+1.0-lpix);
  for (i = 0; i < dim2; i++) lat[i] = y + dy*(i+1.0-bpix);

  switch (bitpix) {
   case 16:
    n = dim1*dim2;
    size = sizeof(WORD);
    map = calloc(n, sizeof(double));
    if (map == NULL) DRPerror("memory allocation failure");
    data = calloc(n, size);
    ip = (WORD *)data;
    if (data == NULL) DRPerror("memory allocation failure");
    else {
      i = readFITSdata(fits, n, size, data);
      if (i == 0) DRPerror("error reading fits data");
    }
    for (j = 0; j < dim2; j++) {
      for (i = 0; i < dim1; i++) {
	k1 = i*dim2+j;
	k2 = j*dim1+i;
#ifdef BYTESWAP
	/* The FITS standard expects the binary words with the MSB first */
	/* We have to swap them on e.g. an Intel CPU */
	swapbytes((char *)&ip[k2], sizeof(WORD));
#endif
	map[k1] = bzero+bscale*ip[k2];
	if ((float)ip[k2] == blank) map[k1] = 0.0; /* blank; */
      }
    }
    break;
   case 32:
    /*
      n = OnScan.NChannel;
      size = sizeof(LONG);
      data = calloc(n, size);
      jp = (LONG *)data;
      if (data == NULL) DRPerror("memory allocation failure");
      else {
      i = readFITSdata(fits, n, size, data);
      if (i == 0) DRPerror("error reading fits data");
      }
      for (i = 0; i < OnScan.NChannel; i++) {
      
#ifdef BYTESWAP
      swapbytes((char *)&jp[i], sizeof(LONG));
#endif
      OnScan.Channels[i] = bzero+bscale*jp[i];
      }
    */
    break;
   default:
    DRPerror("can't read BITPIX = %d format", bitpix);
    break;
  }

  /*
  printf("map = \n");
  for (i = 0; i < dim1; i++) {
    for (j = 0; j < dim2; j++) {
      printf("%lf\n", map[j*dim1+i]);
    }
    printf(" \n");
  }
  */

  fp = fopen("fitsmap.mat","w");
  if (fp == NULL) DRPerror("error opening 'fitsmap.mat'");
  savemat(fp, NUMBER, lon_name,    1, dim1, 0, lon, NULL);
  savemat(fp, NUMBER, lat_name, dim2,    1, 0, lat, NULL);
  /* savemat(fp, NUMBER, map_name, dim1, dim2, 0, map, NULL); */
  savemat(fp, NUMBER, map_name, dim2, dim1, 0, map, NULL);
  fclose(fp);

  free(data);
  free(map);
  DRPinfo("map read from file '%s'", scanname);

  exit(0);
}
