#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "drp.h"
#include "options.h"

static char scanname[MAXNAMLEN+1] = "";

char PROGRAM_NAME[] = "getsest";
char description[] = "get scan in DRP/SEST format";
char required[] = "[scanname]";
struct _options options[] = {
{ "-help",			"Print out this message" },
{ "-file filename",             "specify full filename" },
{NULL, NULL }};

float FHp2IEEE(float);
double DHp2IEEE(double);

#ifdef BYTESWAP
void swapbytes(char *p, int n)
{
    char swap;
    int i;

    for (i = 0; i < n/2; i++) {
	swap = p[i];
	p[i] = p[n-1-i];
	p[n-1-i] = swap;
    }
}
#endif

void getChars(FILE *tar, char *item, int size) {
    int i, c;
    char *next = item;

    for (i = 0; i < size; i++) {
        c = getc(tar);
        next[i] = c;
    }
}

short getShort(FILE *tar) {
    int i, c;
    short item;
    char *next = (char *)&item;

    for (i = 0; i < sizeof(short); i++) {
        c = getc(tar);
        next[i] = c;
    }
#ifdef BYTESWAP
    swapbytes((char *)&item, sizeof(short));
#endif
    return item;
}

int getLong(FILE *tar) {
    int i, c;
    int item;
    char *next = (char *)&item;

    for (i = 0; i < sizeof(int); i++) {
        c = getc(tar);
        /* printf("getLong: [%d] %d %x\n", i, c, c); */
        next[i] = c;
    }
#ifdef BYTESWAP
    swapbytes((char *)&item, sizeof(int));
#endif
    /* printf("getLong: %d\n", item); */

    return item;
}


float getFloat(FILE *tar) {
    int i, c;
    float item;
    char *next = (char *)&item;

    for (i = 0; i < sizeof(float); i++) {
        c = getc(tar);
        next[i] = c;
    }
#ifdef BYTESWAP
    swapbytes((char *)&item, sizeof(float));
#endif
    item = FHp2IEEE(item);

    return item;
}

double getDouble(FILE *tar) {
    int i, c, *m;
    double item;
    char *next = (char *)&item;

    for (i = 0; i < sizeof(double); i++) {
        c = getc(tar);
        next[i] = c;
    }
#ifdef BYTESWAP
    swapbytes((char *)&item, sizeof(double));
#endif

#ifdef BYTESWAP
    m = (int *)next;
    m[0] ^= m[1];
    m[1] ^= m[0];
    m[0] ^= m[1];
#endif
    item = DHp2IEEE(item);
#ifdef BYTESWAP
    m = (int *)next;
    m[0] ^= m[1];
    m[1] ^= m[0];
    m[0] ^= m[1];
#endif

    return item;
}

void ParseOpts(int *pargc, char ***pargv)
{
    char *opt, *optarg;

    opt = (*pargv)[0] + 1;

    if (!strcmp(opt, "help")) {
	Help();
	exit(0);
    }
    if (!strcmp(opt, "file")) {
	GetOption(&optarg, pargc, pargv);
	strcpy(scanname, optarg);
	return;
    }
    Syntax(**pargv);
}

SCAN OnScan;

#define TPOWER  1
#define BSWITCH 2
#define LSWITCH 3
#define SSWITCH 4 
#define FSWITCH 5
#define FOLDED  6

int main(int argc, char *argv[])
{
    FILE *tar;
    int i;

    GetOpts(argc, argv);

    if (scanname[0] == '\0') {
	if (argv[1] == NULL) {
	    Syntax("");
	    exit(EX_ARGSBAD);
	} else {
	    strcpy(scanname, argv[1]);
	}
    }

    tar = fopen (scanname,"r");
    if (tar == NULL) DRPerror("can't open file '%s'", scanname);

    getc(tar); getc(tar);   /* skip two bytes */
    /* printf("sizeof(short)  %ld\n", sizeof(short)); */
    /* printf("sizeof(int)    %ld\n", sizeof(int)); */
    /* printf("sizeof(long)   %ld\n", sizeof(long)); */
    /* printf("sizeof(float)  %ld\n", sizeof(float)); */
    /* printf("sizeof(double) %ld\n", sizeof(double)); */

    memset((char *)&OnScan, 0, HEADER);

    OnScan.Version   = getShort(tar);
    OnScan.NChannel  = getShort(tar);
    OnScan.Ctrl      = getShort(tar);
    OnScan.ScanNo    = getShort(tar);
    OnScan.SubScan   = getShort(tar);
    OnScan.Year      = getShort(tar);
    OnScan.Month     = getShort(tar);
    OnScan.Day       = getShort(tar);
    OnScan.UTHour    = getShort(tar);
    OnScan.UTMin     = getShort(tar);
    OnScan.UTSec     = getShort(tar);
    OnScan.STHour    = getShort(tar);
    OnScan.STMin     = getShort(tar);
    OnScan.STSec     = getShort(tar);
    OnScan.CSystem   = getShort(tar);
    OnScan.ObsMode   = getShort(tar);
    OnScan.Backend   = getShort(tar);
    OnScan.Frontend  = getShort(tar);
    OnScan.MapX      = getShort(tar);
    OnScan.MapY      = getShort(tar);

    printf("file now at %lx\n", ftell(tar));
    OnScan.JulDate   = getLong(tar);

    printf("file now at %lx\n", ftell(tar));
    getChars(tar, OnScan.Name,     12);
    getChars(tar, OnScan.Project,   4);
    getChars(tar, OnScan.Observer, 16);
    getChars(tar, OnScan.Program,  16);
    getChars(tar, OnScan.Molecule, 18);

    OnScan.AirTemp   = getFloat(tar);
    OnScan.Pressure  = getFloat(tar);
    OnScan.Humidity  = getFloat(tar);
    OnScan.Equinox   = getFloat(tar);
    OnScan.EquiNow   = getFloat(tar);
    OnScan.Longitude = getFloat(tar);
    OnScan.Latitude  = getFloat(tar);
    OnScan.Long2000  = getFloat(tar);
    OnScan.Lat2000   = getFloat(tar);
    OnScan.LMapOff   = getFloat(tar);
    OnScan.BMapOff   = getFloat(tar);
    OnScan.Azimuth   = getFloat(tar);
    OnScan.Elevation = getFloat(tar);
    OnScan.AzOffset  = getFloat(tar);
    OnScan.ElOffset  = getFloat(tar);
    OnScan.AzMapOff  = getFloat(tar);
    OnScan.ElMapOff  = getFloat(tar);
    OnScan.AzPointg  = getFloat(tar);
    OnScan.ElPointg  = getFloat(tar);
    OnScan.AzErrAve  = getFloat(tar);
    OnScan.ElErrAve  = getFloat(tar);
    OnScan.AzErrRms  = getFloat(tar);
    OnScan.ElErrRms  = getFloat(tar);
    OnScan.GalLong   = getFloat(tar);
    OnScan.GalLat    = getFloat(tar);
    OnScan.VHel      = getFloat(tar);
    OnScan.VLsr      = getFloat(tar);
    OnScan.Axial     = getFloat(tar);
    OnScan.Shift     = getFloat(tar);
    OnScan.VTilt     = getFloat(tar);
    OnScan.HTilt     = getFloat(tar);
    OnScan.Tcal      = getFloat(tar);
    OnScan.Tsys      = getFloat(tar);
    OnScan.Trec      = getFloat(tar);
    OnScan.Tau       = getFloat(tar);
    OnScan.dBl       = getFloat(tar);
    OnScan.IntTime   = getFloat(tar);
    OnScan.RefCorr   = getFloat(tar);
    OnScan.ParAngle  = getFloat(tar);
    OnScan.PosAngle  = getFloat(tar);
    OnScan.StepX     = getFloat(tar);
    OnScan.StepY     = getFloat(tar);

    OnScan.Bandwidth = getDouble(tar);
    OnScan.RestFreq  = getDouble(tar);
    OnScan.SkyFreq   = getDouble(tar);
    OnScan.FirstIF   = getDouble(tar);
    OnScan.FreqThrow = getDouble(tar);
    OnScan.FreqRes   = getDouble(tar);
    OnScan.VSource   = getDouble(tar);
    OnScan.VelRes    = getDouble(tar);

    for (i = 0; i < 10; i++) OnScan.work[i] = getFloat(tar);
    for (i = 0; i < 31; i++) OnScan.flag[i] = getShort(tar);

    getc(tar); getc(tar);   /* skip two bytes */
    getc(tar); getc(tar);   /* skip two bytes */

    memset((char *)OnScan.Channels, 0, MAXDATA);
    for (i = 0; i < OnScan.NChannel; i++) { 
        OnScan.Channels[i] = getFloat(tar);
    }

    OnScan.Pressure *= 1.3333; /* Torr -> hPa */
    switch (abs(OnScan.ObsMode)) {
      case 0:
        OnScan.ObsMode = FOLDED;
        break;
      case 1:
        OnScan.ObsMode = TPOWER;
        break;
      case 2:
        OnScan.ObsMode = FSWITCH;
        break;
      case 3:
        OnScan.ObsMode = BSWITCH;
        break;
      case 4:
        OnScan.ObsMode = BSWITCH;
        break;
      case 5:
        OnScan.ObsMode = LSWITCH;
        break;
      default:
        OnScan.ObsMode = 0;
        break;
    }

    OnScan.Version = (short)DRPVERSION;

    /* SEST scans have map offsets included in source coordinates,
       we don't want that. */
    OnScan.Latitude -= OnScan.BMapOff;
    OnScan.Longitude -= OnScan.LMapOff/cos(OnScan.Latitude);

    PutScan(&OnScan);
    DRPinfo("scan read from file '%s'", scanname);

    exit(0);
}
