#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "plplot.h"
#include "metadefs.h"
#include "pdf.h"
#include "drp.h"
#include "options.h"

/* top level declarations */

static PLINT	xold, yold;	/* Endpoint of prev line plotted */
static int	no_page;	/* Flag to skip next new page */
static FILE	*MetaFile, *ps;
static char     *FileName, *FileNameOut;
static PLFLT	xdpi, ydpi;
static PLINT	xwid, ywid, xoff, yoff;
static PLINT	nopause;
static char     postscript[10] = "ps";
static char	*devname;
static char	mf_magic[40], mf_version[40];
static U_SHORT	dum_ushort;
static U_CHAR	orient, plr_orient;
static float	aspect = 0.0, plr_aspect = 0.0;
static int	plr_orientset, plr_aspectset /* , plr_colset */;
static PLINT	packx=1, packy=1;
static U_SHORT	lpbpxmi, lpbpxma, lpbpymi, lpbpyma;

static U_SHORT	xmin = 0;
static U_SHORT	xmax = PLMETA_X_OLD;
static U_SHORT	ymin = 0;
static U_SHORT	ymax = PLMETA_Y_OLD;
static PLINT	xlen, ylen;

static float	pxlx = PIXEL_RES_X_OLD;
static float	pxly = PIXEL_RES_Y_OLD;

static PLFLT	dev_xpmm, dev_ypmm;
static PLINT	dev_xmin, dev_xmax, dev_ymin, dev_ymax;
/* static PLINT	plr_col; */
static PLFLT	vpxmin, vpxmax, vpxlen, vpymin, vpymax, vpylen;
static PLFLT	mar=0.0, jx=0.5, jy=0.5;

void plsfile(FILE *);

#define DO_ASP_SCALING

char PROGRAM_NAME[] = "hardcopy";
char description[] = "produce hardcopy of latest graphics operation";
char required[] = "";
struct _options options[] = {
{ "-help",		"Print out this message" },
{ "-dev name",		"Output device name"},
{ "-file name",		"Output file name"},
{ "-geo geom",		"X window size, in pixels (e.g. -geo 400x400)" },
{ "-a aspect",		"Plot aspect ratio" },
{ "-mar margin",	"Total fraction of page to reserve for margins" },
{ "-ori orient",	"Plot orientation (0=landscape, 1=portrait)" },
{ "-jx number",		"Justification of plot on page in x (0.0 to 1.0)" },
{ "-jy number",		"Justification of plot on page in y (0.0 to 1.0)" },
{ "-px number",		"Plots per page in x" },
{ "-py number",		"Plots per page in y" },
{ "-np",		"No pause between pages" },
/* { "-color",		"Which color to plot with" }, */
{NULL, NULL }};

void ParseOpts(int *pargc, char ***pargv)
{
    char *opt, *optarg, *field;

    opt = (*pargv)[0] + 1;

    if (!strcmp(opt, "help")) {
	Help();
	exit(EX_SUCCESS);
    }

/* device */

    if (!strcmp(opt, "dev")) {
	GetOption(&optarg, pargc, pargv);
	devname = optarg;
	return;
    }

/* output file */

    if (!strcmp(opt, "file")) {
	GetOption(&optarg, pargc, pargv);
	FileNameOut = optarg;
	return;
    }

/* Override aspect ratio */

    if (!strcmp(opt, "a")) {
	GetOption(&optarg, pargc, pargv);
	plr_aspect = atof(optarg);
	plr_aspectset = 1;
	return;
    }

/* Set margin factor -- total fraction of page to reserve at edge (includes
   contributions at both sides). */

    if (!strcmp(opt, "mar")) {
	GetOption(&optarg, pargc, pargv);
	mar = atof(optarg);
	return;
    }

/* Set justification in x (0.0 < jx < 1.0). jx = 0.5 (centered) is default */

    if (!strcmp(opt, "jx")) {
	GetOption(&optarg, pargc, pargv);
	jx = atof(optarg);
	return;
    }

/* Set justification in y (0.0 < jy < 1.0). jy = 0.5 (centered) is default */

    if (!strcmp(opt, "jy")) {
	GetOption(&optarg, pargc, pargv);
	jy = atof(optarg);
	return;
    }

/* Override orientation */

    if (!strcmp(opt, "ori")) {
	GetOption(&optarg, pargc, pargv);
	plr_orient = atoi(optarg);
	plr_orientset = 1;
	return;
    }

/* Pack in x */

    if (!strcmp(opt, "px")) {
	GetOption(&optarg, pargc, pargv);
	packx = atoi(optarg);
	return;
    }

/* Pack in y */

    if (!strcmp(opt, "py")) {
	GetOption(&optarg, pargc, pargv);
	packy = atoi(optarg);
	return;
    }

/* No pause between pages */

    if (!strcmp(opt, "np")) {
	nopause++;
	return;
    }

/* Set plot color */
/*
    if (!strcmp(opt, "color")) {
	GetOption(&optarg, pargc, pargv);
	plr_col = atoi(optarg);
	plr_colset = 1;
	return;
    }
*/
/* Geometry for output window (e.g. 400x400+100+0), note offsets don't work
   correctly at present. */

    if (!strcmp(opt, "geo")) {
	GetOption(&optarg, pargc, pargv);

	field = strtok(optarg, "x");
	if (field == NULL)
	    return;
	xwid = atoi(field);

	field = strtok(NULL, "+");
	if (field == NULL)
	    return;
	ywid = atoi(field);

	field = strtok(NULL, "+");
	if (field == NULL)
	    return;
	xoff = atoi(field);

	field = strtok(NULL, "+");
	if (field == NULL)
	    return;
	yoff = atoi(field);

	return;
    }
    Syntax(**pargv);
}

/*  #ifndef sun */
/*  FILE *popen(char *, char *); */
/*  #endif */

/* prototypes for functions in this file. */

static U_CHAR getcommand( void );
static void process_next( U_CHAR );
static void plr_init( void );
static void plresc_rgb( void );
static void plresc_ancol( void );
static int ReadHeader(void);
/* static void check_alignment( FILE * ); */

/*
* Exit codes 
*/

#define	EX_SUCCESS	0		/* success! */
#define	EX_ARGSBAD	1		/* invalid args */
#define	EX_BADFILE	2		/* invalid filename */

static int ReadHeader()
{
    char tag[80];

    /* Read label field of header to make sure file is a PLPLOT metafile */
    plm_rd(read_header(MetaFile, mf_magic));
    if (strcmp(mf_magic, PLPLOT_HEADER)) {
	fprintf(stderr, "Not a PLPLOT metafile!\n");
	return (1);
    }

    /* Read version field of header. */
    plm_rd(read_header(MetaFile, mf_version));
    if (strcmp(mf_version, PLPLOT_VERSION) > 0) {
	fprintf(stderr, "Error: incapable of reading metafile version %s.\n", mf_version);
	return (1);
    }

    /* Return if metafile older than version 1992a (no tagged info). */
    if (strcmp(mf_version, "1992a") < 0) {
	return (0);
    }

    /* Read tagged initialization info. */
    /* This is an easy way to guarantee backward compatibility. */
    for (;;) {
	plm_rd(read_header(MetaFile, tag));
	if (*tag == '\0') break;
	if (!strcmp(tag, "xmin")) {
	    plm_rd(read_2bytes(MetaFile, &xmin));
	    continue;
	}
	if (!strcmp(tag, "xmax")) {
	    plm_rd(read_2bytes(MetaFile, &xmax));
	    continue;
	}
	if (!strcmp(tag, "ymin")) {
	    plm_rd(read_2bytes(MetaFile, &ymin));
	    continue;
	}
	if (!strcmp(tag, "ymax")) {
	    plm_rd(read_2bytes(MetaFile, &ymax));
	    continue;
	}
	if (!strcmp(tag, "pxlx")) {
	    plm_rd(read_ieeef(MetaFile, &pxlx));
	    continue;
	}
	if (!strcmp(tag, "pxly")) {
	    plm_rd(read_ieeef(MetaFile, &pxly));
	    continue;
	}
	if (!strcmp(tag, "aspect")) {
	    plm_rd(read_ieeef(MetaFile, &aspect));
	    continue;
	}
	if (!strcmp(tag, "orient")) {
	    plm_rd(read_1byte(MetaFile, &orient));
	    continue;
	}
	fprintf(stderr, "Unrecognized PLPLOT metafile header tag.\n");
	exit (1);
    }
    return (0);
}


/*----------------------------------------------------------------------*\
*  process_next()
*
*  Process a command
\*----------------------------------------------------------------------*/

static void process_next(U_CHAR c)
{
    U_CHAR op;
    U_SHORT percent, count;
    U_SHORT p1, x1, y1, x2, y2;
    PLFLT x1f, y1f, x2f, y2f;
    PLFLT *xf, *yf;
    int i;

    switch ((int) c) {

      case INITIALIZE:
	plr_init();
	break;

      case SWITCH_TO_TEXT:
	/*	pltext(); */
	break;

      case SWITCH_TO_GRAPH:
	/*	plgra(); */
	break;

      case CLEAR:
	plclr();
	break;

      case PAGE:
	if (strcmp(mf_version, "1992a") >= 0) {
	    plm_rd(read_2bytes(MetaFile, &dum_ushort));
	    plm_rd(read_2bytes(MetaFile, &dum_ushort));
	}
	if (no_page) {
	    no_page = 0;
	    break;
	}
	/*	if ((packx > 1 || packy > 1) && page == 1)
		pladv(0);
		else
		plpage(); */

	plvpor(vpxmin, vpxmax, vpymin, vpymax);
	plwind((PLFLT) xmin, (PLFLT) xmax, (PLFLT) ymin, (PLFLT) ymax);
	break;

      case ADVANCE:
	plm_rd(read_2bytes(MetaFile, &dum_ushort));
	plm_rd(read_2bytes(MetaFile, &dum_ushort));
	if (no_page) {
	    no_page = 0;
	    break;
	}
	pladv(0);
	plvpor(vpxmin, vpxmax, vpymin, vpymax);
	plwind((PLFLT) xmin, (PLFLT) xmax, (PLFLT) ymin, (PLFLT) ymax);
	break;

      case NEW_COLOR:
	plm_rd(read_2bytes(MetaFile, &p1));
	plcol(p1);
	break;

      case NEW_WIDTH:
	plm_rd(read_2bytes(MetaFile, &p1));
	plwid(p1);
	break;

      case LINE:
	plm_rd(read_2bytes(MetaFile, &x1));
	plm_rd(read_2bytes(MetaFile, &y1));
	plm_rd(read_2bytes(MetaFile, &x2));
	plm_rd(read_2bytes(MetaFile, &y2));
	if (!orient) {
	    x1f = x1;
	    y1f = y1;
	    x2f = x2;
	    y2f = y2;
	    pljoin(x1f, y1f, x2f, y2f);
	}
	else {
	    x1f = xmin + (y1 - ymin) * xlen / ylen;
	    y1f = ymin + (xmax - x1) * ylen / xlen;
	    x2f = xmin + (y2 - ymin) * xlen / ylen;
	    y2f = ymin + (xmax - x2) * ylen / xlen;
	    pljoin(x1f, y1f, x2f, y2f);
	}
	xold = x2;
	yold = y2;
	break;

      case LINETO:
	plm_rd(read_2bytes(MetaFile, &x2));
	plm_rd(read_2bytes(MetaFile, &y2));
	if (!orient) {
	    x1f = xold;
	    y1f = yold;
	    x2f = x2;
	    y2f = y2;
	    pljoin(x1f, y1f, x2f, y2f);
	}
	else {
	    x1f = xmin + (yold - ymin) * xlen / ylen;
	    y1f = ymin + (xmax - xold) * ylen / xlen;
	    x2f = xmin + (y2 - ymin) * xlen / ylen;
	    y2f = ymin + (xmax - x2) * ylen / xlen;
	    pljoin(x1f, y1f, x2f, y2f);
	}
	xold = x2;
	yold = y2;
	break;

      case ESCAPE:
	plm_rd(read_1byte(MetaFile, &op));
	switch (op) {

	  case PL_SET_RGB:
	    plresc_rgb();
	    break;

	  case PL_ALLOC_NCOL:
	    plresc_ancol();
	    break;

	  case PL_SET_LPB:
	    plm_rd(read_2bytes(MetaFile, &lpbpxmi));
	    plm_rd(read_2bytes(MetaFile, &lpbpxma));
	    plm_rd(read_2bytes(MetaFile, &lpbpymi));
	    plm_rd(read_2bytes(MetaFile, &lpbpyma));
	    break;
	}
      case AREAFILL:
	plm_rd(read_2bytes(MetaFile, &percent));
	plm_rd(read_2bytes(MetaFile, &count));
	xf = calloc((int)count, sizeof(PLFLT));
	yf = calloc((int)count, sizeof(PLFLT));
	for (i = 0; i < (int)count; i++) {
	    plm_rd(read_2bytes(MetaFile, &x1));
	    plm_rd(read_2bytes(MetaFile, &y1));
	    if (!orient) {
		xf[i] = x1;
		yf[i] = y1;
	    }
	    else {
		xf[i] = xmin + (y1 - ymin) * xlen / ylen;
		yf[i] = ymin + (xmax - x1) * ylen / xlen;
	    }
	}
	plarea((PLINT)count, xf, yf, (PLFLT)percent/100.0);
	break;
    }
}

static void plr_init()
{
    float dev_aspect, ratio;
    char *psdev;

    /* Set up misc. parameters */
    plspage(xdpi, ydpi, xwid, ywid, xoff, yoff);
    plspause(!nopause);

    /* Set up for write.  Filter option always honored. */
    if (FileNameOut != NULL) {
	if (!strcmp(FileNameOut, "-")) plsfile(stdout);
	else                           plsfnam(FileNameOut);
    }

    /* Start up plplot */
    if (devname == postscript) {
	if ((psdev = getenv("PSDEVICE")) != NULL) ps = popen(psdev, "w");
	else                                      ps = popen(PSDEVICE, "w");
	if (ps == NULL) {
	    fprintf(stderr, "%s -- can't pipe output to PostScript printer\n",
		    PROGRAM_NAME);
	    exit(1);
	}
	plsfile(ps);
    }
    plstart(devname, packx, packy);
    pladv(0);

    /* The orientation set from the command line overrides any the user may have
       set before initializing plplot. */
    if (plr_orientset) orient = plr_orient;

    /* Aspect ratio scaling */
    /* If the user has not set the aspect ratio in the code via plsasp() it
       will be zero, and is set to the natural ratio of the metafile coordinate
       system.  The aspect ratio set from the command line overrides this. */

    if (aspect == 0.0) aspect = ((ymax - ymin)/pxly) / ((xmax - xmin)/pxlx);

    if (plr_aspectset) aspect = plr_aspect;

    /* if (plr_colset) plcol(plr_col); */

    if (aspect <= 0.) 
      fprintf(stderr, "Error in aspect ratio setting, aspect = %f\n", aspect);

    if (orient)	aspect = 1.0/aspect;

    /* Aspect ratio of output device */
    gphy(&dev_xmin, &dev_xmax, &dev_ymin, &dev_ymax);
    gpixmm(&dev_xpmm, &dev_ypmm);

    dev_aspect = ((dev_ymax - dev_ymin)/dev_ypmm) /
		 ((dev_xmax - dev_xmin)/dev_xpmm);

    if (dev_aspect <= 0.)
      fprintf(stderr, "Error in aspect ratio setting, dev_aspect = %f\n", dev_aspect);

    ratio = aspect / dev_aspect;

    /* This is the default case; come back to here if things mess up */
    vpxlen = 1.0;
    vpylen = 1.0;
    vpxmin = 0.5 - vpxlen/2.;
    vpymin = 0.5 - vpylen/2.;
    vpxmax = vpxmin + vpxlen;
    vpymax = vpymin + vpylen;

    xlen = xmax - xmin;
    ylen = ymax - ymin;

    /* If ratio < 1, you are requesting an aspect ratio (y/x) less than the natural
       aspect ratio of the output device, and you will need to reduce the length
       in y correspondingly.  Similarly, for ratio > 1, x must be reduced. */

    /* Note that unless the user overrides, the default is to *preserve* the
       aspect ratio of the original device (plmeta output file).  Thus you
       automatically get all physical coordinate plots to come out correctly. */

#ifdef DO_ASP_SCALING
    if (ratio <= 0)
      fprintf(stderr, "Error in aspect ratio setting, ratio = %f\n", ratio);
    else if (ratio < 1)
      vpylen = ratio;
    else
      vpxlen = 1./ratio;

    if (mar > 0.0 && mar < 1.0) {
	vpxlen *= (1.0 - mar);
	vpylen *= (1.0 - mar);
    }

    vpxmin = MAX(0., jx - vpxlen/2.0);
    vpxmax = MIN(1., vpxmin + vpxlen);
    vpxmin = vpxmax - vpxlen;

    vpymin = MAX(0., jy - vpylen/2.0);
    vpymax = MIN(1., vpymin + vpylen);
    vpymin = vpymax - vpylen;
#endif

    plvpor(vpxmin, vpxmax, vpymin, vpymax);
    plwind((PLFLT) xmin, (PLFLT) xmax, (PLFLT) ymin, (PLFLT) ymax);

    no_page++;		/* a kludge */
}

static U_CHAR getcommand(void)
{
    U_CHAR c;

    plm_rd(read_1byte(MetaFile, &c));
    return c;
}

static void plresc_rgb()
{
    float red, green, blue;
    U_SHORT ired, igreen, iblue;

    plm_rd(read_2bytes(MetaFile, &ired));
    plm_rd(read_2bytes(MetaFile, &igreen));
    plm_rd(read_2bytes(MetaFile, &iblue));

    red   = (double) ired   / 65535.;
    green = (double) igreen / 65535.;
    blue  = (double) iblue  / 65535.;

    plrgb((PLFLT) red, (PLFLT) green, (PLFLT) blue);
}

static void plresc_ancol()
{
    U_CHAR icolor;
    char name[80];

    plm_rd(read_1byte(MetaFile, &icolor));
    plm_rd(read_header(MetaFile, name));

    plancol(icolor, name);
}

/*
static void check_alignment(FILE *file)
{
    U_CHAR c;

    plm_rd(read_1byte(file, &c));
    if (c != END_OF_FIELD)
	plexit("Metafile alignment problem");
}
*/

int main(int argc, char *argv[])
{
    U_CHAR c;
    static char metaname[MAXNAMLEN+1];

    GetOpts(argc, argv);
    if (FileName == NULL) {
	strcpy(metaname, getenv("HOME"));
	strcat(metaname, METAFILE);
	FileName = metaname;
    }
    if (devname == NULL)  devname = postscript;

    if (!strcmp(FileName, "-"))
	MetaFile = stdin;

    else if ((MetaFile = fopen(FileName, BINARY_READ)) == NULL) {
	fprintf(stderr, "Unable to open the requested metafile.\n");
	exit(EX_BADFILE);
    }

    /* Check validity of header */
    if (ReadHeader()) exit(EX_BADFILE);

    /* Read & process metafile commands. */
    for (;;) {
	c = getcommand();
	if (c == CLOSE) {
	    if ((c = fgetc(MetaFile)) == (U_CHAR)EOF) break;
	}
	process_next(c);
    }

/* Done */

    (void) fclose(MetaFile);
    plend();
    exit(EX_SUCCESS);
}
