#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "drp.h"  
#include "options.h"

#define H_SHORT 0
#define H_LONG  1
#define MIXER  003
#define VELSYS 014
#define MAPDEF 060

#define LSR 0
#define HEL 1
#define GEO 2

#define TPOWER  1
#define BSWITCH 2
#define LSWITCH 3
#define SSWITCH 4 
#define FSWITCH 5
#define FOLDED  6

#define DESCRIPTIVE 1
#define HORIZONTAL  2

static int list = H_SHORT;

char PROGRAM_NAME[] = "header";
char description[] = "display spectrum header";
char required[] = "";
struct _options options[] = {
{ "-help",			"Print out this message" },
{ "-long",                      "produce long header format" },
{NULL, NULL }};

void ParseOpts(int *pargc, char ***pargv)
{
    char *opt;

    opt = (*pargv)[0] + 1;

    if (!strcmp(opt, "help")) {
	Help();
	exit(EX_SUCCESS);
    }
    if (!strcmp(opt, "long")) {
	list = H_LONG;
	return;
    }
    Syntax(**pargv);
}

char *angle(double, int);
  
SCAN OnScan;

#define NAMELEN 12
#define PROJLEN  4
#define OBSRLEN 16
#define PROGLEN 16
#define MOLELEN 18
#define TOTALEN (NAMELEN+PROJLEN+OBSRLEN+PROGLEN+MOLELEN)

#define NAOS 3
#define NREC 3

static char *Aos[NAOS] = { "   ","HRS","LRS"} ;
static char *Rec[NREC] = { "   ","3MM","1MM"} ;

int main (int argc, char *argv[])
{
    int i;
    char *p;

    GetOpts(argc, argv);

    GetScan (&OnScan);
    p = OnScan.Name;
    for (i = 0; i < TOTALEN; i++) { if (p[i] == '\0') p[i] = ' '; }
    switch (list) {
      case H_SHORT: 
	printf("%04d ",OnScan.ScanNo);
	printf("%.8s", OnScan.Name);
	printf(" %02d%02d%02d.%02d%02d", 
		OnScan.Year%100, OnScan.Month, OnScan.Day,
		OnScan.UTHour, OnScan.UTMin);
	printf(" %s", angle((double)OnScan.Longitude,1));
	printf(" %s", angle((double)OnScan.Latitude,0));
	printf("%5d", drpint((double)OnScan.LMapOff*RADTOSEC));
	printf("%5d", drpint((double)OnScan.BMapOff*RADTOSEC));
	printf("%7.0lf", OnScan.RestFreq);
	printf(" %5.0f", OnScan.Tsys);
	printf("%5d\n", (int)(OnScan.IntTime));
	break;
      case H_LONG:   
	DRPinfo("Scan %04d", OnScan.ScanNo);
	printf("        Date ");
	printf("%02d-%02d-%02d", OnScan.Year, OnScan.Month, OnScan.Day);
	printf("      UTC ");
	printf("%02d:%02d:%02d",OnScan.UTHour,OnScan.UTMin,OnScan.UTSec);
	printf("      LST ");
	printf("%02d:%02d:%02d\n",OnScan.STHour,OnScan.STMin,OnScan.STSec);
	printf("------------------------------------------------------");
	printf("------------------------\n");
	printf("Observer '%.16s'", OnScan.Observer);
	printf("    Molecule '%.18s'", OnScan.Molecule);
	printf("    Project '%.4s'\n", OnScan.Project);
	printf("\n");
	printf("Source name '%.12s'\n", OnScan.Name);
	switch (OnScan.CSystem) {
	  case 1: 
	    printf("RA,Dec 2000.0");
	    printf("%s", angle((double)OnScan.Longitude,1));
	    printf(" %s", angle((double)OnScan.Latitude,0));
	    break;
	  case 2: 
	    printf("RA,Dec %6.1f", OnScan.Equinox);
	    printf("%s", angle((double)OnScan.Longitude, 1));
	    printf(" %s", angle((double)OnScan.Latitude, 0));
	    break;
	  case 3: 
	    printf("RA,Dec (mean)");
	    printf("%s", angle((double)OnScan.Longitude, 1));
	    printf(" %s", angle((double)OnScan.Latitude, 0));
	    break;
	  case 4: 
	    printf("RA,Dec (true)");
	    printf("%s", angle((double)OnScan.Longitude, 0));
	    printf(" %s", angle((double)OnScan.Latitude, 0));
	    break;
	  case 5: 
	    printf("Galactic L,B");
	    printf("%s", angle((double)OnScan.Longitude, 2));
	    printf(" %s", angle((double)OnScan.Latitude, 2));
	    break;
	  case 6: 
	    printf("Azim., Elev. ");
	    printf("%s", angle((double)OnScan.Longitude, 0));
	    printf(" %s", angle((double)OnScan.Latitude, 0));
	    break;
	  default: 
	    printf("RA,Dec %6.1f", OnScan.Equinox);
	    printf("%s", angle((double)OnScan.Longitude, 1));
	    printf(" %s", angle((double)OnScan.Latitude, 0));
	    break;
	}
	printf(" Desc. offsets ");
	printf("%s", angle((double)OnScan.LMapOff, 0));
	printf(" %s\n", angle((double)OnScan.BMapOff, 0));
	printf("Azim., Elev. ");
	printf("%s", angle((double)(OnScan.Azimuth),0));
	printf(" %s", angle((double)OnScan.Elevation,0));
	printf(" Hor. offsets  ");
	printf("%s", angle((double)OnScan.AzMapOff,0));
	printf(" %s\n", angle((double)OnScan.ElMapOff,0));
	printf("Point. model ");
	printf("%s", angle((double)OnScan.AzPointg,0));
	printf(" %s", angle((double)OnScan.ElPointg,0));
	printf(" Point. corr.  ");
	printf("%s", angle((double)OnScan.AzOffset,0));
	printf(" %s\n", angle((double)OnScan.ElOffset,0));
	printf("\n");
	printf("Rest frequency %10.3lf MHz  ", OnScan.RestFreq);
	switch ((OnScan.Ctrl&VELSYS)>>2) {
	  case LSR: 
	    printf(" Vlsr (source)  ");
	    break;
	  case HEL: 
	    printf(" Vhel (source)  ");
	    break;
	  case GEO:
	    printf(" Vgeo (source)  ");
	    break;
	  default:
	    printf(" Vlsr (source)  ");
	    break;
	}
	printf(" %7.1lf km/s   ", OnScan.VSource);
	printf(" Tsys  %6.1f K\n", OnScan.Tsys);
	printf("Sky frequency  %10.3lf MHz  ", OnScan.SkyFreq);
	printf(" Vel. resolution %7.1lf km/s   ", OnScan.VelRes);
	printf(" Trec  %6.1f K\n", OnScan.Trec);
	printf("Freq. resolution %8.4lf MHz  ", OnScan.FreqRes);
	printf(" Vlsr (telescope) %6.1f km/s   ", OnScan.VLsr);
	printf(" Tau   %6.2f  \n", OnScan.Tau);
	printf("Bandwidth      %10.4lf MHz  ", OnScan.Bandwidth);
	printf(" Vhel (telescope) %6.1f km/s   ", OnScan.VHel);
	printf(" Time  %6.1f s\n", OnScan.IntTime);
	printf("\n");
	if (OnScan.Frontend < 0 || OnScan.Frontend > NREC-1) 
	    OnScan.Frontend = 0;
	printf("Frontend %3s", Rec[OnScan.Frontend]);
	switch (OnScan.Ctrl&MIXER) {
	  case 1: 
	    printf("(A)");
	    break;
	  case 2: 
	    printf("(B)");
	    break;
	  case 3: 
	    printf("(S)");
	    break;
	  default: 
	    printf("   ");
	    break;
	}
	if (OnScan.Backend < 0 || OnScan.Backend > NAOS-1) 
	    OnScan.Backend = 0;
	printf("  Backend %3s", Aos[OnScan.Backend]);
	printf(" %4d Channels      ", OnScan.NChannel);
	switch (OnScan.ObsMode) {
	  case TPOWER: 
	    printf("    Obs.mode       total power\n");
	    break;
	  case BSWITCH: 
	    printf("    Obs.mode       beam switch\n");
	    break;
	  case LSWITCH: 
	    printf("    Obs.mode       load switch\n");
	    break;
	  case SSWITCH: 
	    printf("    Obs.mode        sky switch\n");
	    break;
	  case FSWITCH: 
	    printf("Obs.mode Freq.switch %5.1lf MHz\n", 
		   OnScan.FreqThrow);
	  case FOLDED: 
	    printf("Obs.mode  freq.switch (folded)\n");
	    break;
	  default:
	    printf("    Obs.mode        (unknown) \n");
	    break;
	}
	printf("\n");
	printf("Temperature %5.1f K  ", OnScan.AirTemp);
	printf(" Subreflector position   ");
	if (OnScan.LMapOff == 0.0 && OnScan.BMapOff == 0.0) {
	    OnScan.MapX = 0;
	    OnScan.MapY = 0;
	}
	printf("Map position %d %d ", OnScan.MapX, OnScan.MapY);
	switch ((OnScan.Ctrl&MAPDEF)>>4) {
	  case DESCRIPTIVE: 
	    printf("(des)\n");
	    break;
	  case HORIZONTAL:
	    printf("(hor)\n");
	    break;
	  default: 
	    printf("\n");
	    break;
	}
	printf("Pressure %6.1f hPa  ", OnScan.Pressure);
	printf(" Axial %4.1f Shift %4.1f   ", OnScan.Axial, OnScan.Shift);
	printf("Spacing %s", angle((double)OnScan.StepX, 0));
	printf("%s\n", angle((double)OnScan.StepY, 0));
	printf("Humidity   %6.1f %%  ", OnScan.Humidity);
	printf(" VTilt %4.1f HTilt %4.1f   ", OnScan.VTilt, OnScan.HTilt);
	printf("Position angle      %s\n", angle((double)OnScan.PosAngle, 0));
	break;
    }
    exit(EX_SUCCESS);
}
