/* 
 * convert HP1000 A series floating point format to IEEE format
 *
 * who:        Michael Olberg, Onsala Space Observatory
 * when:       92-May-05
 * version:    1.1  - ported from CONVEX which used 64 bit integers
 *                    to do 'double' conversions.
 *
 * caveats: It is not checked that the HP numbers are valid numbers.
 *          The functions were developped to a read binary file from
 *          an A900 into an HP700 work station under the assumption
 *          that the file contained meaningful data.
 *
 * description of floating point formats:
 *
 * note: IEEE has binary biased exponent, subtract 127 for single
 *       precision and 1023 for double precision. HP has signed exponent.
 *       IEEE uses hidden bit, HP doesn't.
 *
 * single precision: (32 bit)
 *           3          2          1
 *          10987654321098765432109876543210
 *  HP:     SMMMMMMMMMMMMMMMMMMMMMMMEEEEEEEX
 *  IEEE:   SEEEEEEEEMMMMMMMMMMMMMMMMMMMMMMM
 *
 * (S=mantissa sign, M=mantissa, E=exponent, X=exponent sign)
 *
 * calling sequence (from C):
 * conversion may be done 'in place', i.e. argument and result may be 
 * represented by the same variable.
 *
 *    float ieee, hpfloat;
 * 
 *    Convert one floating point number.
 *    ieee = FHp2IEEE(hpfloat);
 *
 *
 * double precision: (64 bit)
 *             6         5         4          3         2         1          
 *          32109876543210987654321098765432 10987654321098765432109876543210
 *  HP:     SMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMM MMMMMMMMMMMMMMMMMMMMMMMMEEEEEEEX
 *  IEEE:   SEEEEEEEEEEEMMMMMMMMMMMMMMMMMMMM MMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMM
 *
 * (S=mantissa sign, M=mantissa, E=exponent, X=exponent sign)
 *
 * calling sequence (from C):
 * conversion may be done 'in place', i.e. argument and result may be 
 * represented by the same variable.
 *
 *
 *    double ieee, hpdble;
 *
 *    ieee = DHp2IEEE(hpdble);
 *
 *
 */

#ifdef DEBUG
#include <stdio.h>
#endif

/* ANSI C style function prototypes.                                         */
float FHp2IEEE(float );
double DHp2IEEE(double );

#define SNGLMAN 0x7fffff00          /* bit mask to isolate HP mantissa       */
#define SNGLEXP 0x000000fe          /* bit mask to isolate HP exponent       */

float FHp2IEEE (float fhp)          /* single precision version              */
{
    static float ieee;
    int ms, ex, es;
    int ma, *m, *hp;

    hp = (int *)&fhp;       
#ifdef DEBUG
    printf("enter FHp2IEEE: %08x\n", hp[0]);
#endif
    if (hp[0] == 0L) return (0.0);  /* return 0.0 for all 0 bits             */
    ms = (hp[0] < 0L);              /* test sign bit                         */
    ma = (hp[0] & SNGLMAN) >> 8;    /* extract mantissa                      */
    ma <<= 1;                       /* 'hide' left-most bit                  */
    ex = (hp[0] & SNGLEXP) >> 1;    /* extract exponent                      */
    es = (hp[0] & 1L);              /* test sign of exponent                 */
    if (es) ex = -((ex^0x7f)+1);    /* form two's complement if negative     */
    ex += 127;                      /* form binary biased exponent           */
    ex--;                           /* . is to the right of hidden bit       */
#ifdef DEBUG
    printf("FHp2IEEE: %d %d %d %d\n", es, ex, ms, ma);
#endif

/* start building result                                                     */
    m = (int *)&ieee;   
    m[0] = 0L;                      /* preset to all 0 bits                  */
    if (ms) {                       /* for negative numbers                  */
        m[0] = 0x80000000;          /* set sign bit                          */
        ma = ~ma;                   /* invert mantissa                       */
    }
    ma &= 0x7ffffe;                 /* mask mantissa, 22 valid bits          */
    m[0] |= (int)(ex)<<23;         /* 'or' with binary biased exponent      */
    m[0] |= ma;                     /* 'or' with mantissa                    */
#ifdef DEBUG
    printf("exit FHp2IEEE: %f\n", ieee);
#endif
    return (ieee);
}

#define DBL0MAN 0x7fffffff          /* bit mask to isolate HP mantissa       */
#define DBL1MAN 0xffffff00          /* same for second word                  */
#define DBLEEXP 0x000000fe          /* bit mask to isolate HP exponent       */

double DHp2IEEE (double dhp)        /* double precision version              */
{
    static double ieee;
    int ms, ex, es;
    int ma[2], tmp, *m, *hp;

    hp = (int *)&dhp;
#ifdef DEBUG
    printf("enter DHp2IEEE: %08x:%08x\n", hp[0], hp[1]);
#endif
    if (hp[0] == 0L 
     && hp[1] == 0L) return (0.0);  /* return 0.0 for all 0 bits             */
    ms = (hp[0] < 0L);              /* test sign bit                         */
    ma[0] = (hp[0] & DBL0MAN);      /* extract mantissa                      */
    ma[1] = (hp[1] & DBL1MAN)>>8;   /* extract mantissa                      */
    ma[1] <<= 1;                    /* 'hide' left-most bit                  */
    ma[1] >>= 3;                    /* IEEE mantissa is three bit shorter    */
    tmp = hp[0] & 0x3ff;            /* isolate 10 bit of first word          */
    ma[0] >>= 10;                   /* shift first word right 10 bit         */
    tmp <<= 22;                     /* move bit to upper 10 bits             */
    ma[1] |= tmp;                   /* 'or' with second word                 */
    ex = (hp[1] & DBLEEXP) >> 1;    /* extract exponent                      */
    es = (hp[1] & 1L);              /* test sign of exponent                 */
    if (es) ex = -((ex^0177)+1);    /* form two's complement if negative     */
    ex += 1023;                     /* form binary biased exponent           */
    ex--;                           /* . is to the right of hidden bit       */
#ifdef DEBUG
    printf("DHp2IEEE: %d %d %d %d %d\n", es, ex, ms, ma[0], ma[1]);
#endif

/* start building result                                                     */
    m = (int *)&ieee;   
    m[0] = m[1] = 0L;               /* preset to all zeroes                  */
    if (ms) {                       /* for negative numbers                  */
        m[0] = 0x80000000;          /* set sign bit                          */
        ma[1] = ~ma[1];             /* invert mantissa                       */
        ma[0] = ~ma[0];             /* invert mantissa                       */
    }
    ma[0] &= 0x000fffff;            /* mask mantissa first word (20 bits)    */
    m[0] |= (int)(ex)<<20;         /* 'or' with binary biased exponent      */
    m[0] |= ma[0];                  /* 'or' with mantissa                    */
    m[1] = ma[1];                   /* take over second word as is           */
#ifdef DEBUG
    printf("exit DHp2IEEE: %lf\n", ieee);
#endif
    return (ieee);
}
