#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <ftw.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include "drp.h"
#include "options.h"

#define NORMAL  0
#define NAMES   1
#define NUMBERS 2

double radian(char *);

static int which = NORMAL;
static int header = 1;
static int tabs = 0;
static int namelen;
static char source[12] = "";
static int offset;
static double offx, offy;
static double freq;

char PROGRAM_NAME[] = "list";
char description[] = "produce one line log messages for scan(s)";
char required[] = "filename(s)";
struct _options options[] = {
{ "-help",			"Print out this message" },
{ "-names",                     "print filenames of scans only" },
{ "-numbers",                   "print scan numbers only" },
{ "-nohead",                    "don't print header lines" },
{ "-tab",                       "produce tab separated output" },
{ "-source name",               "list scans with given source name only" },
{ "-offset offx offy",          "list scans with given source name only" },
{ "-freq mhz"                   "list scans with given frequency only" },
{NULL, NULL }};

void ParseOpts(int *pargc, char ***pargv)
{
    char *opt, *optarg;

    opt = (*pargv)[0] + 1;

    if (!strcmp(opt, "names")) {
	which = NAMES;
	return;
    }
    if (!strcmp(opt, "numbers")) {
	which = NUMBERS;
	return;
    }
    if (!strcmp(opt, "nohead")) {
	header = 0;
	return;
    }
    if (!strcmp(opt, "tab")) {
	tabs = 1;
	return;
    }
    if (!strcmp(opt, "source")) {
	GetOption(&optarg, pargc, pargv);
        strcpy(source, optarg);
	namelen = strlen(source);
	return;
    }
    if (!strcmp(opt, "offset")) {
	GetOption(&optarg, pargc, pargv);
	offx = radian(optarg);
	GetOption(&optarg, pargc, pargv);
	offy = radian(optarg);
	offset = 1;
	return;
    }
    if (!strcmp(opt, "freq")) {
	GetOption(&optarg, pargc, pargv);
	freq = atof(optarg);
	return;
    }
    if (!strcmp(opt, "help")) {
	Help();
	exit(EX_SUCCESS);
    }
    Syntax(**pargv);
}

#define READ 0

#define FALSE 0
#define TRUE 1
  
char *angle(double, int);

SCAN OnScan;
struct stat buf;
static int total;

#define NAMELEN 12

void headerline()
{
    /*                1         2         3         4         5         6 */
    /*       123456789012345678901234567890123456789012345678901234567890 */
    printf ("scan source     date.UTC    longitude     latitude    offsets");
    printf ("   freq  Tsys time\n");
    printf ("------------------------------------------------------");
    printf ("-------------------------\n");
}

void footerline()
{
    DRPinfo("total number of scans listed = %d", total);
}

void list_one(const char *name)
{
  static char *months[] = {"Jan", "Feb", "Mar", "Apr", "May", "Jun", 
			   "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };
 
  if (tabs) {
    printf ("%04d\t", OnScan.ScanNo);
    printf ("%.8s\t", OnScan.Name);
    printf ("%s %02d %02d:%02d %04d\t", 
	    months[OnScan.Month-1], OnScan.Day, OnScan.UTHour, 
	    OnScan.UTMin, OnScan.Year);
    printf ("%s\t", angle((double)OnScan.Longitude,1));
    printf ("%s\t", angle((double)OnScan.Latitude,0));
    printf ("%5d\t", drpint((double)OnScan.LMapOff*RADTOSEC));
    printf ("%5d\t", drpint((double)OnScan.BMapOff*RADTOSEC));
    printf ("%7.0lf\t", OnScan.RestFreq);
    printf ("%5.0f\t", OnScan.Tsys);
    printf ("%5d\t", (int)(OnScan.IntTime));
    printf ("%s\n", name);
  } else {
    printf ("%04d ", OnScan.ScanNo);
    printf ("%.8s", OnScan.Name);
    printf (" %02d%02d%02d.%02d%02d", 
	    OnScan.Year%100, OnScan.Month, OnScan.Day,
	    OnScan.UTHour, OnScan.UTMin);
    printf (" %s", angle((double)OnScan.Longitude,1));
    printf (" %s", angle((double)OnScan.Latitude,0));
    printf ("%5d", drpint((double)OnScan.LMapOff*RADTOSEC));
    printf ("%5d", drpint((double)OnScan.BMapOff*RADTOSEC));
    printf ("%7.0lf", OnScan.RestFreq);
    printf (" %5.0f", OnScan.Tsys);
    printf ("%5d\n", (int)(OnScan.IntTime));
  }
  total++;
}

int open_scan(const char *name, const struct stat *statbuf, int flag)
{
    FILE *sc;
    int h;

    if (flag != FTW_F) return (0);
    if ((statbuf->st_size - HEADER) % sizeof(float)) return (0);

    sc = fopen (name, "r");
    if (sc != NULL) {
	h = fread((char *)&OnScan, 1, HEADER , sc);
	if (h == HEADER) {
	    if ((OnScan.Version == (short int)DRPVERSION)
		|| (OnScan.Version == HEADER+2*OnScan.NChannel)) {
#ifdef DEBUG
		printf("open_scan: %s\n", name);
#endif
		if (namelen && strncmp(source, OnScan.Name, namelen)) 
		  return (0);
		if (freq != 0.0 && fabs(freq - OnScan.RestFreq) > 1.0) 
		  return (0);
#ifdef DEBUG
		if (offset) {
		    printf("offsets: %e %e\n",
			   fabs(offx - OnScan.LMapOff),
			   fabs(offy - OnScan.BMapOff));
		}
#endif
		if (offset && fabs(offx - OnScan.LMapOff) > 1.5e-5) 
		  return (0);
		if (offset && fabs(offy - OnScan.BMapOff) > 1.5e-5) 
		  return (0);

		switch (which) {
		  case NAMES:
		    printf("%s\n", name);
		    break;
		  case NUMBERS:
		    printf("%04d\n", OnScan.ScanNo);
		    break;
		  default:
		    list_one(name);
		    break;
		}
	    }
	}
	fclose(sc);
    }
    return (0);
}

int main(int argc, char *argv[])
{
    GetOpts(argc, argv);
    if (which != NORMAL) header = 0;
    if (tabs) header = 0;

    if (argc == 1) {
	Syntax("");
	exit(EX_ARGSBAD);
    }

    if (header) headerline();
#ifdef DEBUG
    printf("offset: %f %f\n", offx, offy);
#endif
    while (--argc) {
	if (stat(*++argv, &buf)) continue;

	if ((buf.st_mode & S_IFMT) != S_IFDIR) {
	    if (buf.st_size >= HEADER) {
		if (open_scan(*argv, &buf, FTW_F)) {
		    continue;
		}
	    }
	} else {
	    ftw(*argv, open_scan, 15);
	}
    }

    if (header) footerline();

    if (total) exit(EX_SUCCESS); /* we found at least one scan */
    else       exit(EX_ARGSBAD);
}
