#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "drp.h"
#include "options.h"

#define MAXRANGE 10

#define MOMENTS    1
#define DISTRIBUTE 2 
#define STATISTICS 4

int what = MOMENTS;
int xUnit = VELOCITY;
double xMin = 0.0, xMax = 0.0;
double clip = 0.0;

char PROGRAM_NAME[] = "moment";
char description[] = "calculate max, integral, centroid and equ.width";
char required[] = "";
struct _options options[] = {
{ "-help",		"Print out this message" },
{ "-stat",              "calculate min, max, mean and rms" },
{ "-dist",              "calculate mean, rms, skewness and kurtosis" },
{ "-vel v1 v2",		"select velocity range" },
{ "-freq f1 f2",	"select frequency range" },
{ "-chan c1 c2",	"select channel range" },
{ "-clip level",	"select noise level for clipping" },
{NULL, NULL }};

void ParseOpts(int *pargc, char ***pargv)
{
    char *opt, *optarg;

    opt = (*pargv)[0] + 1;

    if (!strcmp(opt, "help")) {
	Help();
	exit(0);
    }
    if (!strcmp(opt, "vel")) {
	GetOption(&optarg, pargc, pargv);
	xMin = atof(optarg);
	GetOption(&optarg, pargc, pargv);
	xMax = atof(optarg);
	xUnit = VELOCITY;
	return;
    }
    if (!strcmp(opt, "dist")) {
	what = DISTRIBUTE;
	return;
    }
    if (!strcmp(opt, "stat")) {
	what = STATISTICS;
	return;
    }
    if (!strcmp(opt, "freq")) {
	GetOption(&optarg, pargc, pargv);
	xMin = atof(optarg);
	GetOption(&optarg, pargc, pargv);
	xMax = atof(optarg);
	xUnit = FREQUENCY;
	return;
    }
    if (!strcmp(opt, "chan")) {
	GetOption(&optarg, pargc, pargv);
	xMin = (double)atoi(optarg);
	GetOption(&optarg, pargc, pargv);
	xMax = (double)atoi(optarg);
	xUnit = CHANNELS;
	return;
    }
    if (!strcmp(opt, "clip")) {
	GetOption(&optarg, pargc, pargv);
	clip = atof(optarg);
	return;
    }
    Syntax(**pargv);
}

SCAN OnScan;

int main(int argc, char *argv[])
{
    FILE *lf;
    int FirstCh = 0, LastCh = 0, NCh;
    float offx, offy;
    double integral, mean, centroid, var, sdev, skew, kurt, Tmax, Tmin, ew;
    double Y, X = 0.0;
    int  i;

    GetOpts(argc, argv);

    GetScan(&OnScan);
    if (xMin == 0.0 && xMax == 0.0) {
	FirstCh = 0;
	LastCh = OnScan.NChannel-1;
    } else {
	switch (xUnit) {
	  case VELOCITY:
	    FirstCh = VChannel(xMin, &OnScan);
	    LastCh  = VChannel(xMax, &OnScan);
	    break;
	  case FREQUENCY:
	    FirstCh = FChannel(xMin, &OnScan);
	    LastCh  = FChannel(xMax, &OnScan);
	    break;
	  case CHANNELS:
	    FirstCh = (int)xMin;
	    LastCh  = (int)xMax;
	    break;
	}
    }

    if (FirstCh > LastCh) {
	i = FirstCh;
	FirstCh = LastCh;
	LastCh = i;
    }

    NCh = LastCh - FirstCh + 1;
    if (NCh <= 0) DRPerror("invalid range", 1);

    integral = mean = centroid = ew = 0.0;
    Tmax = Tmin = (double)OnScan.Channels[FirstCh];
    for (i = FirstCh; i <= LastCh; i++) {
	Y = (double)OnScan.Channels[i];
	mean += Y;
	if (fabs(Y) < clip) Y = 0.0;
        if (Y > Tmax) Tmax = Y;
        if (Y < Tmin) Tmin = Y;
	integral += Y;
        switch (xUnit) {
          case VELOCITY:
            X = Velocity(i, &OnScan);
            break;
          case FREQUENCY:
            X = Frequency(i, &OnScan);
            break;
          case CHANNELS:
            X = (double)i;
            break;
	}
	centroid += Y*X; 
    }

    mean = mean/(double)NCh;
    if (integral != 0.0)    centroid /= integral;
    else                    centroid = 0.0;
    if (xUnit == VELOCITY)  integral *= fabs(OnScan.VelRes);
    if (xUnit == FREQUENCY) integral *= fabs(OnScan.FreqRes);
    if (Tmax != 0.0)        ew = integral/Tmax;
    else                    ew = 0.0;
    
    var = skew = kurt = 0.0;
    for (i = FirstCh; i <= LastCh; i++) {
	Y = (double)OnScan.Channels[i] - mean;
	X = Y*Y;
	var += X;
	X *= Y;
	skew += X;
	X *= Y;
	kurt += X;
    }
    var /= (double)(NCh-1);
    sdev = sqrt(var);
    if (var != 0.0) {
	skew /= pow((double)NCh*sdev, 3.0);
	skew *= 100.0;
	kurt /= pow((double)NCh*var, 2.0)-3.0;
	kurt *= 100.0;
    } else {
	skew = 0.0;
	kurt = 0.0;
    }
	
    DRPinfo("results:");
    switch (what) {
      case MOMENTS:
	printf(" Tpeak    = %lf [K]\n", Tmax);
	switch (xUnit) {
	  case VELOCITY: 
	    printf(" Integral = %lf [K*km/s]\n", integral);
	    printf(" Centroid = %lf [km/s]\n", centroid);
	    printf(" Eq.width = %lf [km/s]\n", ew);
	    break;
	  case FREQUENCY: 
	    printf(" Integral = %lf [K*MHz]\n", integral);
	    printf(" Centroid = %lf [MHz]\n", centroid);
	    printf(" Eq.width = %lf [MHz]\n", ew);
	    break;
	  case CHANNELS: 
	    printf(" Integral = %lf [K*channel]\n", integral);
	    printf(" Centroid = %lf [channel]\n", centroid);
	    printf(" Eq.width = %lf [channels]\n", ew);
	    break;
	}
	break;
      case DISTRIBUTE:
	printf(" Mean     = %lf [K]\n", mean);
	printf(" Rms      = %lf [K]\n", sdev);
	printf(" Skewness = %lf [%%]\n", skew);
	printf(" Kurtosis = %lf [%%]\n", kurt);
	break;
      case STATISTICS:
	printf(" Tmin     = %lf [K]\n", Tmin);
	printf(" Tmax     = %lf [K]\n", Tmax);
	printf(" Mean     = %lf [K]\n", mean);
	printf(" Rms      = %lf [K]\n", sdev);
	break;
    }

    offx = OnScan.LMapOff*RADTOMIN;
    offy = OnScan.BMapOff*RADTOMIN;
    if ((lf = fopen("drp.log","a")) != NULL) {
	fprintf(lf, "MOMENT:\t%04d", OnScan.ScanNo);
	fprintf(lf, "\t%10.3f\t%10.3f", offx, offy);

	switch (what) {
	  case MOMENTS:
	    fprintf(lf, "\t%10.3lf\t%10.3lf\t%10.3lf\t%10.3lf\n",
		    Tmax, integral, centroid, ew);
	    break;
	  case DISTRIBUTE:
	    fprintf(lf, "\t%10.3lf\t%10.3lf\t%10.3lf\t%10.3lf\n",
		    mean, sdev, skew, kurt);
	    break;
	  case STATISTICS:
	    fprintf(lf, "\t%10.3lf\t%10.3lf\t%10.3lf\t%10.3lf\n",
		    Tmin, Tmax, mean, sdev);
	    break;
	}
    } else DRPerror("could not append to log file");

    exit(0);
}
