#include <stdio.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include <ctype.h>
#include <sys/stat.h>
#include "drp.h"
#include "astronomy.h"
#include "options.h"

#define WRITE 1
void scanio(char *, int, SCAN *);

char PROGRAM_NAME[] = "mul";
char description[] = "produce DRP scans from MUL data dump";
char required[] = "filename(s)";
struct _options options[] = {
{ "-help",			"Print out this message" },
{ "-dump filename",             "specify filename of data dump" },
{NULL, NULL }};

void ParseOpts(int *pargc, char ***pargv)
{
  char *opt, *optarg;

  opt = (*pargv)[0] + 1;

  if (!strcmp(opt, "help")) {
    Help();
    exit(0);
  }

  Syntax(**pargv);
}

SCAN OnScan;

void KillJunk(char s[])
{
  int i,j;

  for (i = 0, j = 0; i < strlen(s); i++)
    if (isalnum(s[i])) s[j++] = s[i];
  s[j++] = '\0';
}

#define MULCHANNELS 768

int main(int argc, char *argv[])    
{
  FILE *dump;
  int i, scanno, hour, minute, second;
  int nBytes, t[2];
  static int sig[MULCHANNELS], ref[MULCHANNELS];
  long ms;
  struct tm *now;
  time_t clock;
  float utc, az, ell;
  float ra, dec;
  double JD;
  static char scanname[MAXNAMLEN+1];
  char ext[5];

  GetOpts(argc, argv);

  time(&clock);
  now = gmtime(&clock);

  for (scanno = 1; scanno < argc; scanno++) {
    dump = fopen(argv[i], "r");
    if (dump == NULL) 
      DRPerror("couldn't open scan log file '%s'", argv[i]);

    fread(&nBytes, sizeof(int), 1, dump);
    fread(&utc, sizeof(double), 1, dump);
    fread(&t, sizeof(int), 2, dump);
    if (ferror(dump)) DRPerror("error reading data dump");
    if (fread(sig, sizeof(int), MULCHANNELS, dump) != MULCHANNELS) 
      DRPerror("error reading signal data");
    if (fread(ref, sizeof(int), MULCHANNELS, dump) != MULCHANNELS) 
      DRPerror("error reading signal data");
    
    for (i = 0; i < MULCHANNELS; i++) {
      OnScan.Channels[i] = (float)(sig[i]-ref[i])/(float)ref[i];
    }
    
    clock = (time_t)utc;
    
    
    ms = (long)(utc*1000.0);
    hour = (int)(ms/(60*60*1000));
    minute = (int)(ms/(60*1000))%60;
    second = (int)(ms/1000)%60;
    printf("%ld %02d:%02d:%02d\n", ms, hour, minute, second);

    OnScan.NChannel = MAXCHANNELS/4;
    OnScan.Version = DRPVERSION;
    OnScan.ScanNo = scanno;
    OnScan.SubScan = 0;
    OnScan.Year  = now->tm_year+1900;
    OnScan.Month = now->tm_mon+1;
    OnScan.Day   = now->tm_mday;
    JD = djl(OnScan.Year, OnScan.Month, OnScan.Day);
    OnScan.JulDate = (long)(JD+0.5);
    OnScan.UTHour = hour;
    OnScan.UTMin  = minute;
    OnScan.UTSec  = second;
    OnScan.work[0] = utc;
    OnScan.STHour = 0;
    OnScan.STMin  = 0;
    OnScan.STSec  = 0;
    strncpy(OnScan.Name, "Tx Cam      ", 12); 
    OnScan.AirTemp = 280;
    OnScan.Pressure = 1000.0;
    OnScan.Humidity = 75.0;
    OnScan.Equinox = 1950.0;
    OnScan.EquiNow = (float)BEpoch(JD);
    OnScan.Longitude = ra;
    OnScan.Latitude  = dec;
    OnScan.Azimuth   = az;
    OnScan.Elevation = ell;
    OnScan.Tcal    = 250.0;
    OnScan.Tsys    = 500.0;
    OnScan.IntTime = 1.0;
    OnScan.Bandwidth = (double)OnScan.NChannel;
    OnScan.FreqRes   = 0.1;
    OnScan.RestFreq  = 100000.0;
    OnScan.SkyFreq   = 100000.0;
    OnScan.VSource   = 0.0;
    OnScan.VelRes    = -C*1.0e-3*OnScan.FreqRes/OnScan.RestFreq;

    for (i=0; i<OnScan.NChannel; i++) {
      OnScan.Channels[i] = 0.0;
    }

    memset(scanname, '\0', MAXNAMLEN);
    strncpy(scanname, OnScan.Name, 12);
    KillJunk(scanname);
    sprintf (ext, ".%04d", OnScan.ScanNo);
    strcat(scanname, ext);
    scanio(scanname, WRITE, &OnScan);
    DRPinfo("scan written to file '%s'", scanname);
  }

  exit(0);
}
