#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <ctype.h>
#include "drp.h"
#include "options.h"

#ifdef sun
#define memmove memcpy
#endif

static char drspname[MAXNAMLEN+1] = "";

char PROGRAM_NAME[] = "putdrsp";
char description[] = "append scan to a DRAWSPEC data file";
char required[] = "[filename]";
struct _options options[] = {
{ "-help",			"Print out this message" },
{ "-file filename",             "specify DRAWSPEC file name" },
{NULL, NULL }};

void ParseOpts(int *pargc, char ***pargv)
{
    char *opt, *optarg;

    opt = (*pargv)[0] + 1;

    if (!strcmp(opt, "help")) {
	Help();
	exit(0);
    }
    if (!strcmp(opt, "file")) {
	GetOption(&optarg, pargc, pargv);
	strcpy(drspname, optarg);
	return;
    }
    Syntax(**pargv);
}

SCAN OnScan;

#define MAXSCANS 500
#define MAXCHAN 1024

void revb(char *from, int nbytes)
{
    register char *p, *q, c;
    register int i;

    for (i=0, p=from, q=p+nbytes; i<nbytes/2; i++) {
	q--;
	c = *p;
	*p = *q;
	*q = c;
	p++;
    }
}

short int pcint(short int x) 
{
    revb ((char *)&x, sizeof(short int));
    return x;
}

double pcdble(double x)
{
    revb((char *)&x, sizeof(double));
    return x;
}

struct MyFormat {
    char   Source[13], OrigSource[13], date1[9], date2[9], lst[9];
    short  Scan_number, OrigScan_number, Nchan, TheCode, TheOldCode;
    double TC, TS, Int_time, RmsNow;
    double Az, El, Ra, Decl, Gl, Gb, BSum, BDif;
    double Rest_freq, Delta_freq, Center_v, Delta_v;
    short  spectra[MAXCHAN];
    long   next;
} drsp;
/*
 * unfortunately the 'MyFormat' has all record members after drsp.lst
 * aligned on odd byte boundaries. We have to shift all those members
 * by one byte, assuming that the C compiler will have introduced a 
 * pad byte.
 */ 
void align(void)
{
    char *p;
    int len;

    p = drsp.lst+9;
    len = sizeof(struct MyFormat)-(drsp.lst+9-drsp.Source);
    memmove(p, p+1, len);
}

void c2passtr(char *to, char *from, int len)
{
    strncpy(to+1, from, len);
    to[0] = len;
}

void filldrsp(void)
{
    double bzero, bscale;
    float max, min;
    time_t now;
    struct tm *t;
    int i;
    char date[9];


    now = time(NULL); 
    t = gmtime(&now);

    max = min = OnScan.Channels[0];
    for (i = 1; i < OnScan.NChannel; i++) {
	if (max < OnScan.Channels[i]) max = OnScan.Channels[i];
	if (min > OnScan.Channels[i]) min = OnScan.Channels[i];
    }
    bzero = (max+min)/2.0;
    bscale = (max-min)/(2.0*32767.0);
    c2passtr(drsp.Source, OnScan.Name, 12);
    c2passtr(drsp.OrigSource, OnScan.Name, 12);
    sprintf (date, "%02d/%02d/%02d", OnScan.Month, OnScan.Day, OnScan.Year%1900);
    c2passtr(drsp.date1, date, 8);
    sprintf (date, "%02d/%02d/%02d", t->tm_mon+1, t->tm_mday, t->tm_year%1900);
    c2passtr(drsp.date2, date, 8);
    sprintf (date, "%02d:%02d:%02d", OnScan.STHour, OnScan.STMin, OnScan.STSec);
    c2passtr(drsp.lst, date, 8);
    drsp.Scan_number = drsp.OrigScan_number = pcint(OnScan.ScanNo);
    drsp.Nchan = pcint(OnScan.NChannel-1);
    drsp.TheCode = pcint(MAXCHAN-1);
    drsp.TheOldCode = pcint(3);
    drsp.TC = pcdble((double)OnScan.Tcal);
    drsp.TS = pcdble((double)OnScan.Tsys);
    drsp.Int_time = pcdble((double)OnScan.IntTime);
    drsp.RmsNow = pcdble((double)OnScan.Tsys/sqrt(OnScan.Bandwidth*OnScan.IntTime*1.0e6));
    drsp.Az = pcdble((double)OnScan.Azimuth*RADTODEG);
    drsp.El = pcdble((double)OnScan.Elevation*RADTODEG);
    drsp.Ra = pcdble((double)OnScan.Longitude*RADTODEG/15.0);
    drsp.Decl = pcdble((double)OnScan.Latitude*RADTODEG);
    drsp.Gl = pcdble((double)OnScan.GalLong*RADTODEG);
    drsp.Gb = pcdble((double)OnScan.GalLat*RADTODEG);
    drsp.BSum = pcdble(bzero);
    drsp.BDif = pcdble(bscale);
    drsp.Rest_freq = pcdble(OnScan.RestFreq);
    drsp.Delta_freq = pcdble(OnScan.FreqRes);
    drsp.Center_v = pcdble(OnScan.VSource);
    drsp.Delta_v = pcdble(OnScan.VelRes);
    for (i = 0; i < OnScan.NChannel; i++) {
	drsp.spectra[i] = pcint((unsigned int)((OnScan.Channels[i]-bzero)/bscale));
    }
    align();
}

int main(int argc, char *argv[])
{
    FILE *drspfile;
    int len;

    GetOpts(argc, argv);
    if (drspname[0] == '\0') {
	if (argc <= 1) {
	    Syntax("");
	    exit(1);
	} else {
	    strcpy(drspname, argv[1]);
	}
    }

    drspfile = fopen(drspname, "a");
    if (drspfile == NULL) DRPerror("can't append to '%s'", drspname);
    
    GetScan(&OnScan);

    filldrsp();

    len = fwrite((void *)&drsp, sizeof(char), sizeof(short int)*MAXCHAN+195, drspfile);
    if (len != sizeof(short int)*MAXCHAN+195) 
      DRPerror("write error (%d)", len);
    
    DRPinfo("scan written to file '%s'", drspname);

    exit(0);
}
