#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <time.h>
#include <ctype.h>
#include "drp.h"
#include "options.h"
#include "fits.h"
  
static char scanname[MAXNAMLEN+1];

char PROGRAM_NAME[] = "putfits";
char description[] = "store scan as FITS file";
char required[] = "";
struct _options options[] = {
{ "-help",			"Print out this message" },
{ "-file filename",             "specify output filename" },
{NULL, NULL }};

void ParseOpts(int *pargc, char ***pargv)
{
  char *opt, *optarg;

  opt = (*pargv)[0] + 1;

  if (!strcmp(opt, "help")) {
    Help();
    exit(0);
  }

  /* file name */
  if (!strcmp(opt, "file")) {
    GetOption(&optarg, pargc, pargv);
    strcpy(scanname, optarg);
    return;
  }
  Syntax(**pargv);
}

#define FALSE        0
#define TRUE         1

#define BLANK   (-32768)

typedef short int WORD;

extern int used;
extern struct fitskey *kw;
struct fitskey card;
  
void addFITScard(struct fitskey *);
int writeFITSheader(FILE *);
int writeFITSdata(FILE *, int ,int ,void *);
#ifdef BYTESWAP
void swapbytes(char *, int );
#endif

void VoidCard(int index)
{
  card.key = index;
  card.dim = 0;
  addFITScard(&card);
}

void BoolCard(int index, int b)
{
  card.key = index;
  card.dim = 0;
  card.val.l = (long)b;
  addFITScard(&card);
}

void CharCard(int index, int dim, char *s)
{
  int i, len;

  card.key = index;
  card.dim = dim;
  strncpy(card.val.str, s, 16);
  len = strlen(card.val.str);
  if (len < 8) {
    for (i = len; i < 8; i++) card.val.str[i] = ' ';
    card.val.str[8] = '\0';
  }
  addFITScard(&card);
}

void LongCard(int index, int dim, long l)
{
  card.key = index;
  card.dim = dim;
  card.val.l = l;
  addFITScard(&card);
}

void RealCard(int index, int dim, double d)
{
  card.key = index;
  card.dim = dim;
  card.val.d = d;
  addFITScard(&card);
}

SCAN OnScan;

#define VELSYS 0x14

int main(int argc, char *argv[])
{
  FILE *fits;
  static WORD data[MAXCHANNELS];
  char scannum[10], *path;
  char string[20];
  float max, min, Y;
  double bs, bz;
  int i, n, size;
  struct tm *now;
  time_t clock;
  

  GetOpts(argc, argv);

  GetScan(&OnScan);
  sprintf(scannum, "%ddrp", OnScan.ScanNo);

  if (scanname[0] == '\0') {
    if ((path = getenv("FITSAREA")) != NULL) {
      strcpy(scanname, path);
      if (path[strlen(path)-1] != '/') strcat(scanname, "/");
      strcat(scanname, scannum);
    } else {
      strcpy(scanname, scannum);
    }
    strcat(scanname, ".fits");
  }
  fits = fopen(scanname, "w");        /* open the file for reading  */
  if (fits == NULL) DRPerror("can't open file '%s'", scanname);
  used = 0;

  BoolCard(KW_SIMPLE, TRUE);
  LongCard(KW_BITPIX, 0, 16);
  LongCard(KW_NAXIS, 0, 3);
  LongCard(KW_NAXIS, 1, (long)OnScan.NChannel);
  LongCard(KW_NAXIS, 2, 1);
  LongCard(KW_NAXIS, 3, 1);
    
  CharCard(KW_CTYPE, 1, "FREQ");
  RealCard(KW_CRPIX, 1, (double)(CenterCh(&OnScan)+1));
  RealCard(KW_CRVAL, 1, 0.0);
  RealCard(KW_CDELT, 1, OnScan.FreqRes*1.0e6);

  CharCard(KW_CTYPE, 2, "RA---SIN");
  RealCard(KW_CRPIX, 2, 0.0);
  RealCard(KW_CRVAL, 2, OnScan.Longitude*RADTODEG);
  RealCard(KW_CDELT, 2, OnScan.LMapOff*RADTODEG);
  RealCard(KW_CROTA, 2, OnScan.PosAngle*RADTODEG);

  CharCard(KW_CTYPE, 3, "DEC--SIN");
  RealCard(KW_CRPIX, 3, 0.0);
  RealCard(KW_CRVAL, 3, OnScan.Latitude*RADTODEG);
  RealCard(KW_CDELT, 3, OnScan.BMapOff*RADTODEG);
  RealCard(KW_CROTA, 3, OnScan.PosAngle*RADTODEG);

  max = min = OnScan.Channels[0];
  for (i = 1; i < (int)OnScan.NChannel; i++) {
    Y = OnScan.Channels[i];
    if (Y > max) max = Y;
    if (Y < min) min = Y;
  }
  bz = (double)(0.5*(max+min));
  bs = (double)((max-bz)/32767);

  CharCard(KW_BUNIT, 0, "K");
  RealCard(KW_BSCALE, 0, bs);
  RealCard(KW_BZERO, 0, bz);
  RealCard(KW_DATAMAX, 0, (double)max);
  RealCard(KW_DATAMIN, 0, (double)min);
  RealCard(KW_BLANK, 0, (double)BLANK);

  /* sprintf(string, "%02d/%02d/%02d", 
	  OnScan.Day, OnScan.Month, OnScan.Year%100); */
  sprintf(string, "%04d-%02d-%02d", 
	  OnScan.Year, OnScan.Month, OnScan.Day);
  CharCard(KW_DATE_OBS, 0, string);
  time(&clock);
  now = localtime(&clock);
  /* sprintf(string,"%02d/%02d/%02d",
	  now->tm_mday, now->tm_mon+1, now->tm_year%100); */
  sprintf(string, "%04d-%02d-%02d", 
	  now->tm_year+1900, now->tm_mon+1, now->tm_mday);
  CharCard(KW_DATE, 0, string);
  sprintf(string,"%02d:%02d:%02d",
	  OnScan.UTHour, OnScan.UTMin, OnScan.UTSec);
  CharCard(KW_UTC, 0, string);
  sprintf(string,"%02d:%02d:%02d",
	  OnScan.STHour, OnScan.STMin, OnScan.STSec);
  CharCard(KW_LST, 0, string);

  strncpy(string, OnScan.Name, 12); string[12] = '\0';
  CharCard(KW_OBJECT, 0, string);
  CharCard(KW_ORIGIN, 0, "DRP");
  RealCard(KW_EQUINOX, 0, (double)OnScan.Equinox);
  RealCard(KW_AZIMUTH, 0, (double)OnScan.Azimuth*RADTODEG);
  RealCard(KW_ELEVATIO, 0, (double)OnScan.Elevation*RADTODEG);
  RealCard(KW_AZOFF, 0, (double)OnScan.AzMapOff*RADTODEG);
  RealCard(KW_ELOFF, 0, (double)OnScan.ElMapOff*RADTODEG);
  RealCard(KW_AZCORR, 0, (double)OnScan.AzOffset*RADTODEG);
  RealCard(KW_ELCORR, 0, (double)OnScan.ElOffset*RADTODEG);
  RealCard(KW_RESTFREQ, 0, OnScan.RestFreq*1.0e6);
  switch ((OnScan.Ctrl&VELSYS) >> 2) {
   case 0:
    RealCard(KW_VELO_LSR, 0, OnScan.VSource*1.0e3);
    break;
   case 1:
    RealCard(KW_VELO_HEL, 0, OnScan.VSource*1.0e3);
    break;
   case 2:
    RealCard(KW_VELO_GEO, 0, OnScan.VSource*1.0e3);
    break;
   default:
    RealCard(KW_VELO_LSR, 0, OnScan.VSource*1.0e3);
    break;
  }
  RealCard(KW_DELTAV, 0, OnScan.VelRes*1.0e3);
  RealCard(KW_TSYS, 0, (double)OnScan.Tsys);
  RealCard(KW_OBSTIME, 0, (double)OnScan.IntTime);
  RealCard(KW_SCAN, 0, (double)OnScan.ScanNo);
  RealCard(KW_TAU, 0, (double)OnScan.Tau);
  RealCard(KW_DBLOAD, 0, (double)OnScan.dBl);
  strncpy(string, OnScan.Observer, 16); string[16] = '\0';
  CharCard(KW_OBSERVER, 0, string);
  strncpy(string, OnScan.Molecule, 16); string[16] = '\0';
  CharCard(KW_LINE, 0, string);
  RealCard(KW_TOUTSIDE, 0, (double)OnScan.AirTemp);
  RealCard(KW_PRESSURE, 0, (double)OnScan.Pressure*100.0);
  RealCard(KW_HUMIDITY, 0, (double)OnScan.Humidity/100.0);
  VoidCard(KW_END);

  writeFITSheader(fits);
  free(kw);
    
  for (i = 0; i < OnScan.NChannel; i++) {
    data[i] = (WORD)(((double)OnScan.Channels[i]-bz)/bs);
#ifdef BYTESWAP
    /* The FITS standard expects the binary words with the MSB first */
    /* We have to swap them on e.g. an Intel CPU */
    swapbytes((char *)&data[i], sizeof(WORD));
#endif
  }

  n = (int)OnScan.NChannel;
  size = sizeof(WORD);
  i = writeFITSdata(fits, n, size, (void *)data);

  DRPinfo("scan written to file '%s'", scanname);

  exit(0);
}
