#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include "drp.h"
#include "options.h"

int xUnit = VELOCITY;
static char scanname[MAXNAMLEN+1];

char PROGRAM_NAME[] = "putmatlab";
char description[] = "save work area as MATLAB binary data file";
char required[] = "";
struct _options options[] = {
{ "-help",			"Print out this message" },
{ "-file filename",             "specify output filename" },
{ "-vel",		        "save velocity as x-axis (default)" },
{ "-freq",	                "save frequency as x-axis" },
{ "-chan",	                "save channel contents only" },
{NULL, NULL }};

void ParseOpts(int *pargc, char ***pargv)
{
    char *opt, *optarg;

    opt = (*pargv)[0] + 1;

    if (!strcmp(opt, "help")) {
	Help();
	exit(0);
    }

    if (!strcmp(opt, "vel")) {
	xUnit = VELOCITY;
	return;
    }
    if (!strcmp(opt, "freq")) {
	xUnit = FREQUENCY;
	return;
    }
    if (!strcmp(opt, "chan")) {
	xUnit = CHANNELS;
	return;
    }

    /* file name */
    if (!strcmp(opt, "file")) {
	GetOption(&optarg, pargc, pargv);
	strcpy(scanname, optarg);
	return;
    }
    Syntax(**pargv);
}

#define READ 0
#define NAMELEN 12
#define MINIHEADER 10

SCAN OnScan;

#ifdef BYTESWAP
#define DEFAULTTYPE 0
#else
#define DEFAULTTYPE 1000
#endif

#define STRING (DEFAULTTYPE + 1)
#define NUMBER (DEFAULTTYPE)

/* define names and arrays for MATLAB variables */
static char source_name[] = "source";
static double source[NAMELEN+1];

static char header_name[] = "header";
static double header[MINIHEADER];

static char T_name[] = "T";
static double x[MAXCHANNELS];
static double T[MAXCHANNELS];

/*
 * savemat - C language routine to save a matrix in a MAT-file.
 *
 * Here is an example that uses 'savemat' to save two matrices to disk,
 * the second of which is complex:
 *
 *	FILE *fp;
 *	double xyz[1000], ar[1000], ai[1000];
 *	fp = fopen("foo.mat","wb");
 *	savemat(fp, 2000, "xyz", 2, 3, 0, xyz, (double *)0);
 *	savemat(fp, 2000, "a", 5, 5, 1, ar, ai);
 *      fclose(fp);
 *
 * Author J.N. Little 11-3-86
 */

typedef struct {
    long type;   /* type */
    long mrows;  /* row dimension */
    long ncols;  /* column dimension */
    long imagf;  /* flag indicating imag part */
    long namlen; /* name length (including NULL) */
} Fmatrix;

void savemat(FILE *fp, int type, char *pname, int mrows, int ncols, 
	int imagf, double *preal, double *pimag)
/* FILE *fp;       File pointer */
/* int type;       Type flag: Normally 0 for PC, 1000 for Sun, Mac, and	 */
/*		   Apollo, 2000 for VAX D-float, 3000 for VAX G-float    */
/*		   Add 1 for text variables.	 */
/*		   See LOAD in reference section of guide for more info. */
/* int mrows;      row dimension */ 
/* int ncols;      column dimension */
/* int imagf;	   imaginary flag (0: real array, 1: complex array) */
/* char *pname;    pointer to matrix name */
/* double *preal;  pointer to real data */
/* double *pimag;  pointer to imag data */
{
    int mn;
    Fmatrix x;
	
    x.type = type;
    x.mrows = mrows;
    x.ncols = ncols;
    x.imagf = imagf;
    x.namlen = strlen(pname) + 1;
    mn = x.mrows * x.ncols;

    fwrite(&x, sizeof(Fmatrix), 1, fp);
    fwrite(pname, sizeof(char), (int)x.namlen, fp);
    fwrite(preal, sizeof(double), mn, fp);
    if (imagf) {
	fwrite(pimag, sizeof(double), mn, fp);
    }
}

int main(int argc, char *argv[])
{
    FILE *fp;
    char defname[10];
    int i, n;
  
    GetOpts(argc, argv);
    GetScan(&OnScan);
    sprintf(defname, "scan%04d", OnScan.ScanNo);

    if (scanname[0] == '\0') {
	strcpy(scanname, defname);
	strcat(scanname, ".mat");
    }

    fp = fopen(scanname,"w");
    if (fp == NULL) DRPerror("error opening '%s'", scanname);

    for (i=0; i<NAMELEN; i++) {
	source[i] = (double)OnScan.Name[i];
	if (OnScan.Name[i] == '\0') break;
    }
    source[NAMELEN] = 0.0;
    n = i;
    savemat(fp, STRING, source_name, 1, n, 0, source, (double *)0);

    header[0] = (double)OnScan.Longitude*180.0/PI;    /* longitude (deg)    */
    header[1] = (double)OnScan.Latitude*180.0/PI;     /* latitude (deg)     */
    header[2] = (double)OnScan.LMapOff*180.0/PI;      /* offset long. (deg) */
    header[3] = (double)OnScan.BMapOff*180.0/PI;      /* offset lat. (deg)  */
    header[4] = (double)OnScan.Tsys;                  /* Tsys (K)           */
    header[5] = (double)OnScan.IntTime;               /* integ. time (s)    */
    header[6] = OnScan.RestFreq/1.0e3;                /* rest freq. (GHz)   */
    header[7] = OnScan.VSource;                       /* velocity   (km/s)  */
    header[8] = (double)OnScan.Azimuth*180.0/PI;      /* longitude (deg)    */
    header[9] = (double)OnScan.Elevation*180.0/PI;    /* latitude (deg)     */
    savemat(fp, NUMBER, header_name, MINIHEADER, 1, 0, header, (double *)0);

    n = OnScan.NChannel;
    switch (xUnit) {
      case VELOCITY:
	for (i=0; i<n; i++) x[i] = Velocity(i, &OnScan);
	break;
      case FREQUENCY:
	for (i=0; i<n; i++) x[i] = Frequency(i, &OnScan);
	break;
    }
    for (i=0; i<n; i++)	T[i] = (double)OnScan.Channels[i];

    switch (xUnit) {
      case VELOCITY:
      case FREQUENCY:
	savemat(fp, NUMBER, T_name, n, 1, 1, x, T);
	break;
      case CHANNELS:
	savemat(fp, NUMBER, T_name, n, 1, 0, T, (double *)0);
	break;
    }

    fclose(fp);

    DRPinfo("scan written to file '%s'", scanname);

    exit (0);
}
