#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "drp.h"
#include "options.h"

static double red = 0.0;
static int unit = CHANNELS;

char PROGRAM_NAME[] = "redres";
char description[] = "interpolate to different channel spacing";
char required[] = "";
struct _options options[] = {
{ "-help",	   "Print out this message" },
{ "-factor f",     "specify redressing factor > 0.0" }, 
{ "-vel dv",       "specify new velocity resolution" }, 
{ "-freq df",      "specify new frequency resolution" }, 
{NULL, NULL }};

void ParseOpts(int *pargc, char ***pargv)
{
    char *opt, *optarg;

    opt = (*pargv)[0] + 1;

    if (!strcmp(opt, "help")) {
	Help();
	exit(0);
    }
    if (!strcmp(opt, "factor")) {
	GetOption(&optarg, pargc, pargv);
	red = atof(optarg);
	unit = CHANNELS;
	return;
    }
    if (!strcmp(opt, "vel")) {
	GetOption(&optarg, pargc, pargv);
	red = atof(optarg);
	unit = VELOCITY;
	return;
    }
    if (!strcmp(opt, "freq")) {
	GetOption(&optarg, pargc, pargv);
	red = atof(optarg);
	unit = FREQUENCY;
	return;
    }
    Syntax(**pargv);
}

static float wr[MAXCHANNELS], wi[MAXCHANNELS];
  
SCAN OnScan;

int main(int argc, char *argv[])
{
    int i, l, u;
    double p, cc, a, b;
  
    GetOpts(argc, argv);

    GetScan(&OnScan);
    switch (unit) {
      case VELOCITY:
	red = fabs(red/OnScan.VelRes);
	break;
      case FREQUENCY:
	red = fabs(red/OnScan.FreqRes);
	break;
      default:
	break;
    }
    if (red < 1.0) DRPerror("invalid reduction factor (%lf)", red);

    /* determine cubic spline coefficients for spectrum */
    wi[0] = wr[0] = 0.0;
    for (i = 1; i < OnScan.NChannel-1; i++) {
	p = 0.5*OnScan.Channels[i-1]+2.0;
	if (p != 0.0) {
	    wr[i] = -0.5/p;
	    wi[i] = (3.0*(OnScan.Channels[i+1]-2.0*OnScan.Channels[i]
			  +OnScan.Channels[i-1])-0.5*wi[i-1])/p;
	} else wi[i] = wr[i] = 0.0;
    }
    wi[i] = wr[i] = 0.0;
    for (i = OnScan.NChannel-1; i > 0; i--) wr[i-1] = wr[i-1]*wr[i]+wi[i-1];

    /* save old spectrum in 'wi' for interpolation */
    for (i = 0; i < OnScan.NChannel; i++) wi[i] = OnScan.Channels[i];

    /* we want to keep the center channel even in the redressed spectrum */
    cc = (double)CenterCh(&OnScan);
    cc -= floor(cc/red)*red;

    i = 0;
    while (cc <= (double)(OnScan.NChannel-1)) {
	/* interpolation will be between channels l and u of old spectrum */
	l = (int)floor(cc);
	u = (int)ceil(cc);

	if (l == u) {
	    OnScan.Channels[i] = wi[l]; /* just copy old value */
	} else {
	    a = ceil(cc)-cc;
	    b = cc-floor(cc);
	    /* perform cubic spline interpolation */
	    OnScan.Channels[i] = a*(wi[l]+(a*a-1)*wr[l]/6)
                                +b*(wi[u]+(b*b-1)*wr[u]/6);
	}
	i++;       /* count channels in redressed spectrum */
	cc += red;
    }

    /* update scan header */
    OnScan.NChannel = i;
    OnScan.FreqRes *= red;
    OnScan.VelRes  *= red;
    OnScan.Bandwidth = OnScan.FreqRes*OnScan.NChannel;

    PutScan (&OnScan);
    exit(0);
}
