#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "plplot.h"
#include "metadefs.h"
#include "pdf.h"
#include "drp.h"
#include "options.h"

/* top level declarations */

static PLINT	xold, yold;	/* Endpoint of prev line plotted */
static int	no_page;	/* Flag to skip next new page */
static FILE	*MetaFile;
static char     *FileName, *FileNameOut;
static PLFLT	xdpi, ydpi;
static PLINT	xwid, ywid, xoff, yoff;
static PLINT	nopause;
static char	mf_magic[40], mf_version[40];
static U_SHORT	dum_ushort;
static U_CHAR	orient, plr_orient;
static float	aspect = 0.0, plr_aspect = 0.0;
static int	plr_orientset, plr_aspectset;
static PLINT	packx=1, packy=1;
static U_SHORT	lpbpxmi, lpbpxma, lpbpymi, lpbpyma;

static U_SHORT	xmin = 0;
static U_SHORT	xmax = PLMETA_X_OLD;
static U_SHORT	ymin = 0;
static U_SHORT	ymax = PLMETA_Y_OLD;
static PLINT	xlen, ylen;

static float	pxlx = PIXEL_RES_X_OLD;
static float	pxly = PIXEL_RES_Y_OLD;

static PLFLT	dev_xpmm, dev_ypmm;
static PLINT	dev_xmin, dev_xmax, dev_ymin, dev_ymax;
static PLFLT	vpxmin, vpxmax, vpxlen, vpymin, vpymax, vpylen;

void plsfile(FILE *);

char PROGRAM_NAME[] = "reshow";
char description[] = "redo graphics operation";
char required[] = "";
struct _options options[] = {
{ "-help",		"Print out this message" },
{NULL, NULL }};

void ParseOpts(int *pargc, char ***pargv)
{
    char *opt;

    opt = (*pargv)[0] + 1;

    if (!strcmp(opt, "help")) {
	Help();
	exit(0);
    }
    Syntax(**pargv);
}

/* prototypes for functions in this file. */

static U_CHAR getcommand( void );
static void process_next( U_CHAR );
static void plr_init( void );
static void plresc_rgb( void );
static void plresc_ancol( void );
static void ReadHeader(void);
/* static void check_alignment( FILE * ); */

/*
* Exit codes 
*/

static void ReadHeader()
{
    char tag[80];

    /* Read label field of header to make sure file is a PLPLOT metafile */
    plm_rd(read_header(MetaFile, mf_magic));
    if (strcmp(mf_magic, PLPLOT_HEADER)) exit(2);

    /* Read version field of header. */
    plm_rd(read_header(MetaFile, mf_version));
    if (strcmp(mf_version, PLPLOT_VERSION) > 0) exit(3);

    /* Return if metafile older than version 1992a (no tagged info). */
    if (strcmp(mf_version, "1992a") < 0) return;

    /* Read tagged initialization info. */
    /* This is an easy way to guarantee backward compatibility. */
    for (;;) {
	plm_rd(read_header(MetaFile, tag));
	if (*tag == '\0') break;
	if (!strcmp(tag, "xmin")) {
	    plm_rd(read_2bytes(MetaFile, &xmin));
	    continue;
	}
	if (!strcmp(tag, "xmax")) {
	    plm_rd(read_2bytes(MetaFile, &xmax));
	    continue;
	}
	if (!strcmp(tag, "ymin")) {
	    plm_rd(read_2bytes(MetaFile, &ymin));
	    continue;
	}
	if (!strcmp(tag, "ymax")) {
	    plm_rd(read_2bytes(MetaFile, &ymax));
	    continue;
	}
	if (!strcmp(tag, "pxlx")) {
	    plm_rd(read_ieeef(MetaFile, &pxlx));
	    continue;
	}
	if (!strcmp(tag, "pxly")) {
	    plm_rd(read_ieeef(MetaFile, &pxly));
	    continue;
	}
	if (!strcmp(tag, "aspect")) {
	    plm_rd(read_ieeef(MetaFile, &aspect));
	    continue;
	}
	if (!strcmp(tag, "orient")) {
	    plm_rd(read_1byte(MetaFile, &orient));
	    continue;
	}
	exit(4);
    }
    return;
}


/*----------------------------------------------------------------------*\
*  process_next()
*
*  Process a command
\*----------------------------------------------------------------------*/

static void process_next(U_CHAR c)
{
    U_CHAR op;
    U_SHORT percent, count;
    U_SHORT p1, x1, y1, x2, y2;
    PLFLT x1f, y1f, x2f, y2f;
    PLFLT *xf, *yf;
    int i;

    switch ((int) c) {

    case INITIALIZE:
	plr_init();
	break;

    case SWITCH_TO_TEXT:
/*	pltext(); */
	break;

    case SWITCH_TO_GRAPH:
/*	plgra(); */
	break;

    case CLEAR:
	plclr();
	break;

    case PAGE:
	if (strcmp(mf_version, "1992a") >= 0) {
	    plm_rd(read_2bytes(MetaFile, &dum_ushort));
	    plm_rd(read_2bytes(MetaFile, &dum_ushort));
	}
	if (no_page) {
	    no_page = 0;
	    break;
	}

	plvpor(vpxmin, vpxmax, vpymin, vpymax);
	plwind((PLFLT) xmin, (PLFLT) xmax, (PLFLT) ymin, (PLFLT) ymax);
	break;

    case ADVANCE:
	plm_rd(read_2bytes(MetaFile, &dum_ushort));
	plm_rd(read_2bytes(MetaFile, &dum_ushort));
	if (no_page) {
	    no_page = 0;
	    break;
	}
	pladv(0);
	plvpor(vpxmin, vpxmax, vpymin, vpymax);
	plwind((PLFLT) xmin, (PLFLT) xmax, (PLFLT) ymin, (PLFLT) ymax);
	break;

    case NEW_COLOR:
	plm_rd(read_2bytes(MetaFile, &p1));
	plcol(p1);
	break;

    case NEW_WIDTH:
	plm_rd(read_2bytes(MetaFile, &p1));
	plwid(p1);
	break;

    case LINE:
	plm_rd(read_2bytes(MetaFile, &x1));
	plm_rd(read_2bytes(MetaFile, &y1));
	plm_rd(read_2bytes(MetaFile, &x2));
	plm_rd(read_2bytes(MetaFile, &y2));
	if (!orient) {
	    x1f = x1;
	    y1f = y1;
	    x2f = x2;
	    y2f = y2;
	    pljoin(x1f, y1f, x2f, y2f);
	}
	else {
	    x1f = xmin + (y1 - ymin) * xlen / ylen;
	    y1f = ymin + (xmax - x1) * ylen / xlen;
	    x2f = xmin + (y2 - ymin) * xlen / ylen;
	    y2f = ymin + (xmax - x2) * ylen / xlen;
	    pljoin(x1f, y1f, x2f, y2f);
	}
	xold = x2;
	yold = y2;
	break;

    case LINETO:
	plm_rd(read_2bytes(MetaFile, &x2));
	plm_rd(read_2bytes(MetaFile, &y2));
	if (!orient) {
	    x1f = xold;
	    y1f = yold;
	    x2f = x2;
	    y2f = y2;
	    pljoin(x1f, y1f, x2f, y2f);
	}
	else {
	    x1f = xmin + (yold - ymin) * xlen / ylen;
	    y1f = ymin + (xmax - xold) * ylen / xlen;
	    x2f = xmin + (y2 - ymin) * xlen / ylen;
	    y2f = ymin + (xmax - x2) * ylen / xlen;
	    pljoin(x1f, y1f, x2f, y2f);
	}
	xold = x2;
	yold = y2;
	break;

    case ESCAPE:
	plm_rd(read_1byte(MetaFile, &op));
	switch (op) {

	case PL_SET_RGB:
	    plresc_rgb();
	    break;

	case PL_ALLOC_NCOL:
	    plresc_ancol();
	    break;

	case PL_SET_LPB:
	    plm_rd(read_2bytes(MetaFile, &lpbpxmi));
	    plm_rd(read_2bytes(MetaFile, &lpbpxma));
	    plm_rd(read_2bytes(MetaFile, &lpbpymi));
	    plm_rd(read_2bytes(MetaFile, &lpbpyma));
	    break;
	}
      case AREAFILL:
	plm_rd(read_2bytes(MetaFile, &percent));
	plm_rd(read_2bytes(MetaFile, &count));
	xf = calloc((int)count, sizeof(PLFLT));
	yf = calloc((int)count, sizeof(PLFLT));
	for (i = 0; i < (int)count; i++) {
	    plm_rd(read_2bytes(MetaFile, &x1));
	    plm_rd(read_2bytes(MetaFile, &y1));
	    if (!orient) {
		xf[i] = x1;
		yf[i] = y1;
	    }
	    else {
		xf[i] = xmin + (ymax - y1) * xlen / ylen;
		yf[i] = ymin + (x1 - xmin) * ylen / xlen;
	    }
	}
	plarea((PLINT)count, xf, yf, (PLFLT)percent/100.0);
	break;
    }
}

static void plr_init()
{
    float dev_aspect /*, ratio */;

    /* Set up misc. parameters */
    plspage(xdpi, ydpi, xwid, ywid, xoff, yoff);
    plspause(!nopause);

    /* Set up for write.  Filter option always honored. */
    if (FileNameOut != NULL) {
	if (!strcmp(FileNameOut, "-")) plsfile(stdout);
	else                           plsfnam(FileNameOut);
    }

    /* Start up plplot */
    plstart(PLDEVICE, packx, packy);
    pladv(0);
    plclr(); /* needed for our DRP window driver */

    /* The orientation set from the command line overrides any the user may have
       set before initializing plplot. */
    if (plr_orientset) orient = plr_orient;

    /* Aspect ratio scaling */
    /* If the user has not set the aspect ratio in the code via plsasp() it
       will be zero, and is set to the natural ratio of the metafile coordinate
       system.  The aspect ratio set from the command line overrides this. */

    if (aspect == 0.0) aspect = ((ymax - ymin)/pxly) / ((xmax - xmin)/pxlx);

    if (plr_aspectset) aspect = plr_aspect;

    if (aspect <= 0.) 
      fprintf(stderr,"Error in aspect ratio setting (%f)\n", aspect);

    if (orient)	aspect = 1.0/aspect;

    /* Aspect ratio of output device */
    gphy(&dev_xmin, &dev_xmax, &dev_ymin, &dev_ymax);
    gpixmm(&dev_xpmm, &dev_ypmm);

    dev_aspect = ((dev_ymax - dev_ymin)/dev_ypmm) /
		 ((dev_xmax - dev_xmin)/dev_xpmm);

    if (dev_aspect <= 0.)
      fprintf(stderr, "Error in aspect ratio setting (%f)\n", dev_aspect);

    /* ratio = aspect / dev_aspect; */

    /* This is the default case; come back to here if things mess up */
    vpxlen = 1.0;
    vpylen = 1.0;
    vpxmin = 0.5 - vpxlen/2.;
    vpymin = 0.5 - vpylen/2.;
    vpxmax = vpxmin + vpxlen;
    vpymax = vpymin + vpylen;

    xlen = xmax - xmin;
    ylen = ymax - ymin;

    /* If ratio < 1, you are requesting an aspect ratio (y/x) less than the natural
       aspect ratio of the output device, and you will need to reduce the length
       in y correspondingly.  Similarly, for ratio > 1, x must be reduced. */

    /* Note that unless the user overrides, the default is to *preserve* the
       aspect ratio of the original device (plmeta output file).  Thus you
       automatically get all physical coordinate plots to come out correctly.*/
    plvpor(vpxmin, vpxmax, vpymin, vpymax);
    plwind((PLFLT) xmin, (PLFLT) xmax, (PLFLT) ymin, (PLFLT) ymax);

    no_page++;		/* a kludge */
}

static U_CHAR getcommand(void)
{
    U_CHAR c;

    plm_rd(read_1byte(MetaFile, &c));
    return c;
}

static void plresc_rgb()
{
    float red, green, blue;
    U_SHORT ired, igreen, iblue;

    plm_rd(read_2bytes(MetaFile, &ired));
    plm_rd(read_2bytes(MetaFile, &igreen));
    plm_rd(read_2bytes(MetaFile, &iblue));

    red   = (double) ired   / 65535.;
    green = (double) igreen / 65535.;
    blue  = (double) iblue  / 65535.;

    plrgb((PLFLT) red, (PLFLT) green, (PLFLT) blue);
}

static void plresc_ancol()
{
    U_CHAR icolor;
    char name[80];

    plm_rd(read_1byte(MetaFile, &icolor));
    plm_rd(read_header(MetaFile, name));

    plancol(icolor, name);
}

/*
static void check_alignment(FILE *file)
{
    U_CHAR c;

    plm_rd(read_1byte(file, &c));
    if (c != END_OF_FIELD)
	plexit("Metafile alignment problem");
}
*/

int main(int argc, char *argv[])
{
    U_CHAR c;
    static char metaname[MAXNAMLEN+1];

    strcpy(metaname, getenv("HOME"));
    strcat(metaname, METAFILE);
    FileName = metaname;

    if ((MetaFile = fopen(FileName, BINARY_READ)) == NULL) exit(1);

    /* Check validity of header */
    ReadHeader();

    /* Read & process metafile commands. */
    for (;;) {
	c = getcommand();
	if (c == CLOSE) {
	    if ((c = fgetc(MetaFile)) == (U_CHAR)EOF) break;
	}
	process_next(c);
    }

/* Done */

    (void) fclose(MetaFile);
    plend();
    exit(0);
}
