#include <stdio.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include <ctype.h>
#include <sys/stat.h>
#include "drp.h"
#include "astronomy.h"
#include "options.h"

#define WRITE 1
void scanio(char *, int, SCAN *);

static char logname[MAXNAMLEN+1] = "";

char PROGRAM_NAME[] = "scan";
char description[] = "produce DRP scans from scan log";
char required[] = "";
struct _options options[] = {
{ "-help",			"Print out this message" },
{ "-log filename",              "specify filename of log data" },
{NULL, NULL }};

void ParseOpts(int *pargc, char ***pargv)
{
  char *opt, *optarg;

  opt = (*pargv)[0] + 1;

  if (!strcmp(opt, "log")) {
    GetOption(&optarg, pargc, pargv);
    strcpy(logname, optarg);
    return;
  }
  if (!strcmp(opt, "help")) {
    Help();
    exit(0);
  }

  Syntax(**pargv);
}

SCAN OnScan;

void KillJunk(char s[])
{
  int i,j;

  for (i = 0, j = 0; i < strlen(s); i++)
    if (isalnum(s[i])) s[j++] = s[i];
  s[j++] = '\0';
}

/*
 * calculate year, month, day from Julian Day
 */
void cld(double JD, int *Year, int *Month, int *Day)
{
  double ip, f;
  int z, alpha, a, b, c, d, e;

  f = modf(JD+0.5, &ip);
  z = (int)ip;
  if (z < 2299161) {
    a = z;
  } else {
    alpha = (int)((ip - 1867216.25)/36524.25);
    a = z + 1 + alpha - alpha/4;
  }

  b = a + 1524;
  c = (int)(((double)b-122.1)/365.25);
  d = (int)(365.25*c);
  e = (int)((double)(b-d)/30.6001);

  *Day   = b - d - (int)(30.6001*e);
  *Month = (e < 14 ? e-1 : e-13);
  *Year  = (*Month > 2 ? c - 4716 : c - 4715);
}

void swapbytes(char *p, int n)
{
  char swap;
  int i;

  for (i = 0; i < n/2; i++) {
    swap = p[i];
    p[i] = p[n-1-i];
    p[n-1-i] = swap;
  }
}
    

#define MULCHANNELS 768

int main(int argc, char *argv[])    
{
  FILE *scanlog, *dump;
  int i, n, scanno, year, month, day, hour, minute, second;
  long ms;
  struct tm *now;
  time_t clock;
  static char line[256];
  double utc, ip, JD;
  int nBytes, t[2];
  float azold, elold, aznew, elnew, elr, tickold, ticknew;
  static char scanname[MAXNAMLEN+1];
  static int sig[MULCHANNELS], ref[MULCHANNELS];
  char ext[5];

  GetOpts(argc, argv);
  GetScan (&OnScan);

  time(&clock);
  now = gmtime(&clock);

  if (logname[0] == '\0') 
    DRPerror("scan log file name required");
  scanlog = fopen(logname, "r");
  if (scanlog == NULL) DRPerror("couldn't open scan log file '%s'", logname);

  scanno = 0;
/*   if (fgets(line, 255, scanlog) != NULL) { */
/*     sscanf(line, "%f %f", &ra, &dec); */
/*   } else { */
/*     DRPwarning("unable to read RA, Dec"); */
/*     ra = dec = 0.0; */
/*   } */
/*   printf("RA, Dec %10.6f %10.6f\n", ra, dec); */

  if (fgets(line, 255, scanlog) != NULL) {
    sscanf(line, "%f %f %f %f", &tickold, &azold, &elold, &elr);
  } else {
    DRPwarning("error reading scan log");
    tickold = azold = elold = 0.0;
  }

  if (fgets(line, 255, scanlog) != NULL) {
    sscanf(line, "%f %f %f %f", &ticknew, &aznew, &elnew, &elr);
  } else {
    DRPwarning("error reading scan log");
    ticknew = aznew = elnew = 0.0;
  }

  scanno = 0;
  for (n = 3; n < argc; n++) {
    dump = fopen(argv[n], "r");
    if (dump == NULL) 
      DRPerror("couldn't open scan log file '%s'", argv[n]);

    scanno++;
    fread(&nBytes, sizeof(int), 1, dump);
    fread(&utc, sizeof(double), 1, dump);
    fread(&t, sizeof(int), 2, dump);
#ifdef BYTESWAP
    swapbytes((char *)&utc, sizeof(double));
    swapbytes((char *)&nBytes, sizeof(int));
    swapbytes((char *)&t[0], sizeof(int));
    swapbytes((char *)&t[1], sizeof(int));
#endif
    if (ferror(dump)) DRPerror("error reading data dump");
    if (fread(sig, sizeof(int), MULCHANNELS, dump) != MULCHANNELS) 
      DRPerror("error reading signal data");
    if (fread(ref, sizeof(int), MULCHANNELS, dump) != MULCHANNELS) 
      DRPerror("error reading signal data");
    fclose(dump);

    JD = djl(1970, 1, 1)+utc/SECPERDAY;
    cld(JD, &year, &month, &day);
    utc = modf(utc/86400.0, &ip)*86400.0;

    ms = (long)(utc*1000.0);
    hour = (int)(ms/(60*60*1000));
    minute = (int)(ms/(60*1000))%60;
    second = (int)(ms/1000)%60;
    printf("%04d-%02d-%02d %02d:%02d:%02d\n", 
	   year, month, day, 
	   hour, minute, second);

    OnScan.NChannel = MULCHANNELS;
    for (i = 0; i < MULCHANNELS; i++) {
#ifdef BYTESWAP
      swapbytes((char *)&sig[i], sizeof(int));
      swapbytes((char *)&ref[i], sizeof(int));
#endif
      OnScan.Channels[i] = (float)(sig[i]-ref[i])/(float)ref[i];
    }

    while (ticknew < (float)utc) {
      if (fgets(line, 255, scanlog) != NULL) {
	tickold = ticknew;
	azold   = aznew;
	elold   = elnew;
	sscanf(line, "%f %f %f %f", &ticknew, &aznew, &elnew, &elr);
      } else {
	DRPwarning("error reading scan log");
	break;
      }
    }
    
    /*     OnScan.Version = DRPVERSION; */
    OnScan.ScanNo = scanno;
    /*     OnScan.SubScan = 0; */
    OnScan.Year  = year;
    OnScan.Month = month;
    OnScan.Day   = day;
    JD = djl(OnScan.Year, OnScan.Month, OnScan.Day);
    OnScan.JulDate = (long)(JD+0.5);
    OnScan.UTHour = hour;
    OnScan.UTMin  = minute;
    OnScan.UTSec  = second;
    OnScan.work[0] = utc;
    /*     OnScan.STHour = 0; */
    /*     OnScan.STMin  = 0; */
    /*     OnScan.STSec  = 0; */
    /*     strncpy(OnScan.Name, "mul-scan    ", 12);  */
    /*     OnScan.AirTemp = 280; */
    /*     OnScan.Pressure = 1000.0; */
    /*     OnScan.Humidity = 75.0; */
    /*     OnScan.Equinox = 1950.0; */
    OnScan.EquiNow = (float)BEpoch(JD);
    /*     OnScan.Longitude = ra; */
    /*     OnScan.Latitude  = dec; */
    OnScan.Azimuth   = azold+(utc-tickold)*(aznew-azold)/(ticknew-tickold);
    OnScan.Elevation = elold+(utc-tickold)*(elnew-elold)/(ticknew-tickold);
    printf("old: %10.2f %10.6f %10.6f\n", tickold, azold, elold);
    printf("new: %10.2f %10.6f %10.6f\n", ticknew, aznew, elnew);
    printf("%10.2f %10.6f %10.6f\n", utc, OnScan.Azimuth, OnScan.Elevation);
    /*     OnScan.Tcal    = 250.0; */
    /*     OnScan.Tsys    = 500.0; */
    OnScan.IntTime = (t[0]+t[1])/1000.0;
    OnScan.FreqRes   = 1.0;
    OnScan.Bandwidth = (double)OnScan.NChannel*OnScan.FreqRes;
    /*     OnScan.RestFreq  = 100000.0; */
    /*     OnScan.SkyFreq   = 100000.0; */
    /*     OnScan.VSource   = 0.0; */
    /*     OnScan.VelRes    = -C*1.0e-3*OnScan.FreqRes/OnScan.RestFreq; */

    memset(scanname, '\0', MAXNAMLEN);
    strncpy(scanname, OnScan.Name, 12);
    KillJunk(scanname);
    sprintf (ext, ".%04d", OnScan.ScanNo);
    strcat(scanname, ext);
    scanio(scanname, WRITE, &OnScan);
    DRPinfo("scan written to file '%s'", scanname);
  }

  exit(0);
}
