#include <stdio.h>
#include <string.h>
#include <math.h>
#include "plplot.h"
#include "drp.h"

char *xlabel[] = { 
    "velocity [km/s]", 
    "frequency [MHz]", 
    "channels", 
    "velocity offset [km/s]", 
    "frequency offset [MHz]", 
    "channel offset", 
    NULL 
};

char *ylabel[] = { 
    "T [K]", 
    NULL 
};

extern char xLabel[], yLabel[];
  
#define WINDOW     1
#define VIEWPORT   2
#define RELATIVE 256
  
static float xarray[MAXCHANNELS];
static float yarray[MAXCHANNELS];

SCAN OnScan;
char *angle(double, int);
void Header (SCAN *);

extern int zeroline, histogram, frame, relative, grid, title, label[];
extern char NewTitle[];

void spect(int xUnit, float xMin, float xMax, float yMin, float yMax)
{
    float Y, yStep;
    float VxMin, VxMax, VyMin, VyMax;
    float swap, delta = 0.0, centre = 0.0, scale;
    int FirstCh, LastCh, NCh;
    int i;
  
  
    GetScan(&OnScan);
    FirstCh = 0; 
    LastCh = OnScan.NChannel-1;
  
    switch (xUnit) {
      case VELOCITY:  
	delta = OnScan.VelRes;
	centre = (float)OnScan.VSource;
	break;
      case FREQUENCY:
	delta = OnScan.FreqRes;
	centre = (float)OnScan.RestFreq;
	break;
      case CHANNELS:
	delta = 1.0;
	centre = (float)CenterCh(&OnScan);
	break;
    }

    if (delta < 0.0) {
	swap = xMin; 
	xMin = xMax; 
	xMax = swap;
    }
  
    if (xMin != 0.0 || xMax != 0.0) {
	switch (xUnit) {
	  case VELOCITY:
            FirstCh = VChannel((double)xMin, &OnScan);
            LastCh  = VChannel((double)xMax, &OnScan);
            break;
	  case FREQUENCY:
            FirstCh = FChannel((double)xMin, &OnScan);
            LastCh  = FChannel((double)xMax, &OnScan);
            break;
	  case CHANNELS:
            FirstCh = (int)xMin;
	    if (FirstCh < 0) {
		FirstCh = 0;
		xMin = (float)FirstCh;
	    }
            LastCh  = (int)xMax;
	    if (LastCh > OnScan.NChannel-1) {
		LastCh = OnScan.NChannel-1;
		xMax = (float)LastCh;
	    }
	}
    }
    switch (xUnit) {
      case VELOCITY:
	xMin = (float)Velocity(FirstCh, &OnScan);
	xMax = (float)Velocity(LastCh, &OnScan);
	break;
      case FREQUENCY:
	xMin = (double)Frequency(FirstCh, &OnScan);
	xMax = (double)Frequency(LastCh, &OnScan);
	break;
      case CHANNELS:
	if (xMin == 0.0 && xMax == 0.0) {
	    xMin = 0.0;
	    xMax = (float)(OnScan.NChannel-1);
	}
	break;
    }

    if (FirstCh > LastCh) {
	i = FirstCh;
	FirstCh = LastCh;
	LastCh = i;
    }
  
    NCh = LastCh - FirstCh + 1;
    if (relative) {
	xMin -= centre;
	xMax -= centre;
    }
  
    if (yMin == 0.0 && yMax == 0.0) {
	yMin = OnScan.Channels[FirstCh];
	yMax = OnScan.Channels[FirstCh];
	for (i=1; i<NCh; i++) {
	    Y = OnScan.Channels[FirstCh+i];
	    if (Y < yMin) yMin = Y;
	    if (Y > yMax) yMax = Y;
	}
	yStep = (yMax-yMin)/20.0;
	yMin -= yStep;
	yMax += yStep;
    }

    pladv(0);
    if (OnScan.flag[2] == VIEWPORT) {
	VxMin = OnScan.work[4];
	VxMax = OnScan.work[5];
	VyMin = OnScan.work[6];
	VyMax = OnScan.work[7];
	plvpor((PLFLT)VxMin, (PLFLT)VxMax, (PLFLT)VyMin, (PLFLT)VyMax);
	scale = VyMax-VyMin;
	plsmaj(0.0, scale);
	plsmin(0.0, scale);
	plschr(0.0, scale);
    } else plvsta();

    plwind((PLFLT)xMin, (PLFLT)xMax, (PLFLT)yMin, (PLFLT)yMax);
    if (frame) {
	if (grid) plbox("abcgnst", (PLFLT)0.0, 0, "abcgnstv", (PLFLT)0.0, 0);
	else      plbox("bcnst", (PLFLT)0.0, 0, "bcnstv", (PLFLT)0.0, 0);
	if (!label[0]) {
	    if (!relative) strcpy(xLabel, xlabel[xUnit]);
	    else           strcpy(xLabel, xlabel[xUnit+3]);
	}
	if (!label[1]) strcpy(yLabel, ylabel[0]);
	pllab(xLabel, yLabel," ");
    } else {
	plbox("", (PLFLT)0.0, 0, "", (PLFLT)0.0, 0);
    }	
    if (zeroline) {
	if (yMin < 0.0 && yMax > 0.0) {
	    xarray[0] = xMin;
	    xarray[1] = xMax;
	    yarray[0] = 0.0;
	    yarray[1] = 0.0;
	}
	plline(2, xarray, yarray);
    }

    if (NCh > MAXCHANNELS/2) histogram = 0;
    if (histogram) {
	xarray[0] = xMin;
	yarray[0] = OnScan.Channels[FirstCh];
	for (i=1; i<NCh; i++) {
	    xarray[2*i-1] = xMin + (float)i*delta - delta/2.0;
	    yarray[2*i-1] = OnScan.Channels[FirstCh+i-1];
	    xarray[2*i]   = xarray[2*i-1];
	    yarray[2*i]   = OnScan.Channels[FirstCh+i];
	}
	xarray[2*NCh-1] = xMin + (float)(NCh-1)*delta;
	yarray[2*NCh-1] = OnScan.Channels[LastCh];
	plline(2*NCh, xarray, yarray);
    } else {
	for (i=0; i<NCh; i++) {
	    xarray[i] = xMin + (float)i*delta;
	    yarray[i] = OnScan.Channels[FirstCh+i];
	}
	plline(NCh, xarray, yarray);
    }

    OnScan.flag[0] = xUnit;

    OnScan.flag[1] = WINDOW;
    if (relative) OnScan.flag[1] |= RELATIVE;
    OnScan.work[0] = xMin;
    OnScan.work[1] = xMax;
    OnScan.work[2] = yMin;
    OnScan.work[3] = yMax;

    if (frame) Header(&OnScan);
    PutScan(&OnScan);
}

void Header(SCAN *sc)
{
   float l,b;
   float disp, pos, just;
   int lo,bo;
   char str[80], lon[13], lat[13];
  
   if (title) {
       disp = -1.5; pos = 0.975; just = 1.0;
       plmtex ("t", disp, pos, just, NewTitle); 
   } else {
       l = sc->Longitude;
       b = sc->Latitude;
       lo = drpint((double)sc->LMapOff*RADTOSEC);
       bo = drpint((double)sc->BMapOff*RADTOSEC);
       switch (sc->CSystem) {
	 case 5:		/* galactic coordinates */
	   strcpy (lon, angle((double)l,2));
	   strcpy (lat, angle((double)b,2));
	   break;
	 case 6:		/* horizontal coordinates */
	   strcpy (lon, angle((double)l,0));
	   strcpy (lat, angle((double)b,0));
	   break;
	 default:		/* equatorial coordinates */
	   strcpy (lon, angle((double)l,1));
	   strcpy (lat, angle((double)b,0));
       }
       sprintf (str,"%04d %.12s %s %s (%d,%d)", 
		sc->ScanNo, sc->Name, lon, lat, lo, bo);
       disp = 2.5; pos = 0.0; just = 0.0;
       plmtex ("t", disp, pos, just, str); 
       sprintf (str,"%02d%02d%02d.%02d%02d  %.10s  %9.1lf MHz  %d sec",
		sc->Year-1900, sc->Month, sc->Day, sc->UTHour, sc->UTMin,
		sc->Molecule, sc->RestFreq, (int)sc->IntTime);
       disp = 1.0;
       plmtex ("t", disp, pos, just, str); 
   }
}
