#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include "drp.h"
#include "options.h"

#define MAXITER 100
#define DIM 4
#define EPS 0.01
  
int xUnit = VELOCITY;
float xMin = 0.0, xMax = 0.0;
double Amplitude = 0.0;
double Centre = 0.0;
double Width = 0.0;
double Area = 0.0;
double Tau = 0.0;
int aguess = 0, cguess = 0, wguess = 0, tguess = 0;

int CenterCh(SCAN *);
double matinv(double **, int);
double profile(double, double [], double []);
int VChannel(double , SCAN *);
int FChannel(double , SCAN *);

char PROGRAM_NAME[] = "taufit";
char description[] = "fit profile to HCN hyperfine components";
char required[] = "";
struct _options options[] = {
{ "-help",		"print out this message" },
{ "-vel v1 v2",		"select velocity range" },
{ "-freq f1 f2",	"select frequency range" },
{ "-chan c1 c2",	"select channel range" },
{ "-ampl amplitude",    "guess amplitude of gaussian" },
{ "-centre centre",     "guess centre of gaussian" },
{ "-width width",       "guess width of gaussian" },
{ "-tau t",             "guess optical depth" },
{NULL, NULL }};

void ParseOpts(int *pargc, char ***pargv)
{
    char *opt, *optarg;

    opt = (*pargv)[0] + 1;

    if (!strcmp(opt, "help")) {
	Help();
	exit(0);
    }
    if (!strcmp(opt, "vel")) {
	GetOption(&optarg, pargc, pargv);
	xMin = atof(optarg);
	GetOption(&optarg, pargc, pargv);
	xMax = atof(optarg);
	xUnit = VELOCITY;
	return;
    }
    if (!strcmp(opt, "freq")) {
	GetOption(&optarg, pargc, pargv);
	xMin = atof(optarg);
	GetOption(&optarg, pargc, pargv);
	xMax = atof(optarg);
	xUnit = FREQUENCY;
	return;
    }
    if (!strcmp(opt, "chan")) {
	GetOption(&optarg, pargc, pargv);
	xMin = (double)atoi(optarg);
	GetOption(&optarg, pargc, pargv);
	xMax = (double)atoi(optarg);
	xUnit = CHANNELS;
	return;
    }
    if (!strcmp(opt, "ampl")) {
	GetOption(&optarg, pargc, pargv);
	Amplitude = atof(optarg);
	aguess = 1;
	return;
    }
    if (!strcmp(opt, "centre")) {
	GetOption(&optarg, pargc, pargv);
	Centre = atof(optarg);
	cguess = 1;
	return;
    }
    if (!strcmp(opt, "width")) {
	GetOption(&optarg, pargc, pargv);
	Width = atof(optarg);
	wguess = 1;
	return;
    }
    if (!strcmp(opt, "tau")) {
	GetOption(&optarg, pargc, pargv);
	Tau = atof(optarg);
	tguess = 1;
	return;
    }

    Syntax(**pargv);
}

SCAN OnScan;

int main(int argc, char *argv[])
{
    FILE *lf;
    float offx, offy;
    double a[DIM*DIM], s[DIM*DIM];
    double Vector[DIM], new[DIM], D[DIM], old[DIM];
    double *Array[DIM], *Scale[DIM];
    double Fit, Res;
    double Chiold, Chinew, delta, lamda;
    double X, Y, max;
    int iter, ready, freedom, inner;
    int i, j, l, cc = 0;
    int FirstCh = 0, LastCh = 0;

    GetOpts(argc, argv);

    GetScan (&OnScan);
    if (xMin == 0.0 && xMax == 0.0) {
	FirstCh = 0;
	LastCh = OnScan.NChannel-1;
    } else {
	switch (xUnit) {
	  case VELOCITY:
	    FirstCh = VChannel((double)xMin, &OnScan);
	    LastCh  = VChannel((double)xMax, &OnScan);
	    break;
	  case FREQUENCY:
	    FirstCh = FChannel((double)xMin, &OnScan);
	    LastCh  = FChannel((double)xMax, &OnScan);
	    break;
	  case CHANNELS:
	    FirstCh = (int)xMin;
	    LastCh  = (int)xMax;
	    break;
	}
    }

    if (FirstCh > LastCh) {
	i = FirstCh;
	FirstCh = LastCh;
	LastCh = i;
    }

    max = OnScan.Channels[0];
    for (i = 1; i < OnScan.NChannel; i++) {
	if (OnScan.Channels[i] > max) {
	    max = OnScan.Channels[i];
	    cc = i;
	}
    }
    if (!aguess)  Amplitude = max;
    if (!cguess)  Centre = (double)cc;
    if (!wguess)  Width = 10.0;
    if (!tguess)  Tau = 10.0;

/*
    switch (xUnit) {
      case VELOCITY: 
	if (cguess) Centre = VChannel((double)Centre, &OnScan);
	if (wguess) Width /= fabs(OnScan.VelRes);
	break;
      case FREQUENCY: 
	if (cguess) Centre = FChannel((double)Centre, &OnScan);
	if (wguess) Width /= fabs(OnScan.FreqRes);
	break;
      case CHANNELS: 
	break;
    }
*/  
    if (FirstCh > LastCh) {
	i = FirstCh;
	FirstCh = LastCh;
	LastCh = i;
    }
 
    freedom = LastCh-FirstCh-DIM+1;
    if (freedom <= 0) DRPerror("too few points to fit");
  
    DRPinfo("guess values:");
    printf (" Amplitude = %lf\n", Amplitude);
    printf (" Centre    = %lf\n", Centre);
    printf (" Width     = %lf\n", Width );
    printf (" Opt.depth = %lf\n", Tau );
    printf (" fit between channels %d and %d\n", FirstCh, LastCh);
  
    new[0] = Amplitude;
    new[1] = Centre;
    new[2] = Width;
    new[3] = Tau;
/*
    printf (" new values=%10.3f %10.3f %10.3f %10.3f\n", 
	    new[0], new[1], new[2], new[3]);
*/
    for (i = 0; i < DIM; i++) {
	Array[i] = &a[i*DIM];
	Scale[i] = &s[i*DIM];
    }

    lamda = 0.01;
    iter = 0;
    ready = 0;
    while (!ready) {
	iter++;
	if (iter > MAXITER) break;

	lamda /= 10.0;
	for (i = 0; i < DIM; i++) old[i] = new[i];
	for (i = 0; i < DIM; i++) {
	    Vector[i] = 0.0;
	    for (j = i; j < DIM; j++) {
		Array[i][j] = 0.0;
		Array[j][i] = 0.0;
	    }
	}
  
	Chiold = 0.0;
	for (i = FirstCh; i <= LastCh; i++) {
	    X = Velocity(i, &OnScan);
	    Y = OnScan.Channels[i];
	    Fit = profile(X, new, D);
	    Res = Y-Fit;
	    Chiold += Res*Res;
	    for (j = 0; j < DIM; j++) {
		Vector[j] += Res*D[j];
		for (l = j; l < DIM; l++) {
		    Array[j][l] += D[j]*D[l];
		    Array[l][j] = Array[j][l];
		}
	    }
	}
	Chiold /= (double)freedom;
  
	inner = 0;
	do {
	    inner++;
	    if (inner > MAXITER) DRPerror("no convergence in inner loop");

	    for (i=0; i<DIM; i++) {
		if (Array[i][i] == 0.0) DRPerror("bad initial guesses");
		for (j=0; j<DIM; j++) {
		    Scale[i][j] = Array[i][j]
		      /sqrt(Array[i][i]*Array[j][j]);
		}
		Scale[i][i] = 1+lamda;
	    }
 
	    if (matinv(Scale, DIM) == 0.0) DRPerror("bad initial guesses");
  
	    for (i = 0; i < DIM; i++) {
		new[i] = old[i];
		for (j = 0; j < DIM; j++) {
		    new[i] += Vector[j]*Scale[i][j]
		      /sqrt(Array[i][i]*Array[j][j]);
		}
	    }
  
	    Chinew = 0.0;
	    for (i = FirstCh; i <= LastCh; i++) {
		X = Velocity(i, &OnScan);
		Y = OnScan.Channels[i];
		Fit = profile(X, new, D);
		Res = Y-Fit;
		Chinew += Res*Res;
	    }
	    Chinew /= (double)freedom;
	    if (Chinew > Chiold) lamda *= 10.0;
/*
	    printf (" [%2d:%2d] X(old)=%10.3f X(new)=%10.3f (%10.3f)\n",
		    iter, inner, Chiold, Chinew, lamda);

	    printf (" new values=%10.3f %10.3f %10.3f %10.3f\n", 
		    new[0], new[1], new[2], new[3]);
*/
	} while (Chinew > Chiold);
  
	ready = 1;
	for (i=0; i<DIM; i++) {
	    delta = fabs((old[i]-new[i])/new[i]);
/*	    printf("delta[%d] = %10.4lf\n", i, delta); */
	    if (delta > EPS) ready = 0;
	}
    }

  
    for (i=0; i<OnScan.NChannel; i++) {
	X = Velocity(i, &OnScan);
	OnScan.Channels[i] = profile(X, new, D);
    }
    for (i=0; i<DIM; i++) OnScan.work[i] = new[i];

    if (iter > MAXITER) DRPwarning("no convergence");
    else    DRPinfo("converged after %d iterations", iter);
    printf (" fit results:\n");
    printf (" Amplitude = %f [K]\n", new[0]);
/*
    if (argc == 1) xUnit = VELOCITY;
    switch (xUnit) {
      case CHANNELS:
	printf (" Centre    = %f [channel]\n", new[1]);
	printf (" Width     = %f [channels]\n", new[2]);
	break;
      case VELOCITY:
	new[1] = (new[1]-CenterCh(&OnScan))*OnScan.VelRes+OnScan.VSource;
	new[2] *= fabs(OnScan.VelRes);
	printf (" Centre    = %f [km/s]\n", new[1]);
	printf (" Width     = %f [km/s]\n", new[2]);
	break;
      case FREQUENCY:
	new[1] = (new[1]-CenterCh(&OnScan))*OnScan.FreqRes+OnScan.RestFreq;
	new[2] *= fabs(OnScan.FreqRes);
	printf (" Centre    = %f [MHz]\n", new[1]);
	printf (" Width     = %f [MHz]\n", new[2]);
	break;
    }
*/
    printf (" Centre    = %f [km/s]\n", new[1]);
    printf (" Width     = %f [km/s]\n", new[2]);
    printf (" Opt.depth = %f [neper]\n", new[3]);
    PutTemp (&OnScan);
  
    offx = OnScan.LMapOff*RADTOMIN;
    offy = OnScan.BMapOff*RADTOMIN;
    if ((lf = fopen("drp.log","a")) != NULL) {
	fprintf (lf, "TAUFIT:\t%04d", OnScan.ScanNo);
	fprintf (lf, "\t%10.3f\t%10.3f", offx, offy);
	fprintf (lf,"\t%10.3f\t%10.3f\t%10.3f\t%10.3f\n",
		 new[0], new[1], new[2], new[3]);
    } else DRPerror("could not append to log file");

    exit (0);
}
  
#define W 1.665109222   /* sqrt (4*ln2) */
  
double profile(double X, double Coeff[], double Deriv[])
{
   double arg;
   double sum, d1, d2, tau, c;
   int i;
   static double v[3] = { 3.981, -0.796, -7.962 };
   static double s[3] = { 0.333,  0.555,  0.111 };

   sum = d1 = d2 = 0.0;
   for (i = 0; i < 3; i++) {
       arg = W*(X-Coeff[1]-v[i])/Coeff[2];
       c = s[i]*exp(-arg*arg);
       sum += c;
       d1 += c*(-2.0*arg)*(-W/Coeff[2]);
       d2 += c*(-2.0*arg)*(-arg/Coeff[2]);
   }
   tau = Coeff[3]*sum;
   Deriv[0] = 1.0-exp(-tau);
   Deriv[1] = Coeff[0]*exp(-tau)*Coeff[3]*d1;
   Deriv[2] = Coeff[0]*exp(-tau)*Coeff[3]*d2;
   Deriv[3] = Coeff[0]*exp(-tau)*sum;
/*
   printf("vel %10.3lf: tau = %10.3lf  T = %10.3lf\n", 
	  X, tau, Coeff[0]*Deriv[0]);
*/
   return (Coeff[0]*Deriv[0]);
}
