#include <QLocale>
#include <QMessageBox>
#include <QTableWidgetItem>
#include <QDateTime>
#include <QDate>
#include <QList>
#include <QPen>
#include <QFont>
#include <QIcon>
#include <QFile>
#include <QFileInfo>
#include <QDir>
#include <QString>
#include <QInputDialog>
#include <QProcess>
#include <QByteArray>
#include <QProgressDialog>
#include <QSettings>
#include <QDebug>

#include "mainwindow.h"
#include "ui_mainwindow.h"
#include "garmindialog.h"
#include "statistics.h"
#include "track.h"

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    ui->setupUi(this);
    setWindowIcon(QIcon(":/images/qtrainer.xpm"));
    ui->actionOpen->setEnabled(QFile::exists("/usr/bin/garmin_dump"));
    ui->actionImport->setEnabled(QFile::exists("/usr/bin/garmin_save_runs"));
    lstr = NULL;
    cpt = NULL;
    ipt = NULL;

    QSettings settings("qtrainer", "qtrainer");
    QString dbname = settings.value("database/filename").toString();
    if (dbname.isEmpty()) {
        dbname = QDir::homePath() + "/.qtrainer/forerunner.db";
        settings.setValue("database/filename", dbname);
    }
    qDebug() << "using database:" << dbname;

    qreal myLon = settings.value("home/longitude").toDouble();
    qreal myLat = settings.value("home/latitude").toDouble();
    if (myLon == 0.0 && myLat == 0.0) {
        myLon =  12.070792;
        myLat =  57.503674;
        settings.setValue("home/longitude", myLon);
        settings.setValue("home/latitude",  myLat);
    }
    currentTrack = new Track(QPointF(myLon, myLat), -1);

    QFileInfo fi(dbname);
    QDir dbdir = fi.dir();
    if (!dbdir.exists()) {
        dbdir.mkpath(".");
    }

    db = QSqlDatabase::addDatabase("QSQLITE");

    db.setDatabaseName(dbname);
    db.open();

    bool ok = checkDatabase();
    if (!ok) {
        QMessageBox::critical(0, "QTrainer",
                              "Database file " + dbname + " not "
                              "found! Please see the REAMDE file for "
                              "how to set this up.");
    }

    ui->tableWidget->setColumnCount(3);
    QSqlQuery q = QSqlQuery("SELECT count(*) FROM tracks");
    if (q.first()) {
        noOfTracks = q.value(0).toInt();
        ui->tableWidget->setColumnCount(3);
        ui->tableWidget->setRowCount(noOfTracks);

        QStringList labels;
        labels << tr("date time") << tr("track") << tr("distance");
        ui->tableWidget->setHorizontalHeaderLabels(labels);
        ui->tableWidget->setColumnWidth(0, 160);
        ui->tableWidget->setColumnWidth(1, 300);
        ui->tableWidget->setColumnWidth(2, 80);
    }

    retrieveTable();
    connect(ui->tableWidget->verticalHeader(), SIGNAL(sectionClicked(int)), 
            this, SLOT(trackSelected(int)));
    connect(ui->tableWidget, SIGNAL(cellClicked(int, int)), 
            this, SLOT(editTrack(int, int)));

    mc = new MapControl(QSize(480,480));
    mc->showScale(false);
    // display the MapControl in the application
    QVBoxLayout* layout = new QVBoxLayout;
    layout->addWidget(mc);

    QWidget *w = (QWidget *)ui->mapFrame;
    w->setLayout(layout);

    mapadapter = new GoogleMapAdapter();
    // mapadapter = new OSMMapAdapter();
    l = new MapLayer("Track Layer", mapadapter);
    mc->addLayer(l);
    addZoomButtons();
    mc->setView(currentTrack->centre());
    mc->setZoom(13);

    int width = QApplication::desktop()->width();
    // int height = QApplication::desktop()->height();
    int height = minimumSizeHint().height();
    // int height = ui->statTable->minimumSizeHint().height();
    // qDebug() << "setting dimensions to" << width << "x" << height;
    resize(width-100, height);

    speed = new QwtPlotCurve("speed");
    QPen blue = QPen(QColor(Qt::blue));
    speed->setPen(blue);
    speed->attach(ui->speedPlot);

    pulse = new QwtPlotCurve("pulse");
    QPen green = QPen(QColor(Qt::green));
    // green.setWidth(2);
    pulse->setPen(green);
    pulse->attach(ui->pulsePlot);

    profile = new QwtPlotCurve("profile");
    profile->attach(ui->profilePlot);

    QPen gray = QPen(QColor(Qt::gray));
    grid1 = new QwtPlotGrid();
    grid1->setPen(gray);
    grid1->attach(ui->speedPlot);

    grid2 = new QwtPlotGrid();
    grid2->setPen(gray);
    grid2->attach(ui->pulsePlot);

    grid3 = new QwtPlotGrid();
    grid3->setPen(gray);
    grid3->attach(ui->profilePlot);
    picker1 = new QwtPlotPicker(QwtPlot::xBottom, QwtPlot::yLeft,
                                QwtPicker::PointSelection, 
                                QwtPlotPicker::CrossRubberBand,
                                QwtPicker::AlwaysOn,
                                ui->speedPlot->canvas());
    picker1->setRubberBandPen(QColor(Qt::red));
    picker1->setTrackerMode(QwtPicker::ActiveOnly);
    connect(picker1, SIGNAL(moved(QwtDoublePoint)), 
            this, SLOT(markCurrent(QwtDoublePoint)));
    connect(picker1, SIGNAL(selected(QwtDoublePoint)), 
            this, SLOT(markCurrent(QwtDoublePoint)));

    picker2 = new QwtPlotPicker(QwtPlot::xBottom, QwtPlot::yLeft,
                                QwtPicker::PointSelection, 
                                QwtPlotPicker::CrossRubberBand,
                                QwtPicker::AlwaysOn,
                                ui->pulsePlot->canvas());
    picker2->setRubberBandPen(QColor(Qt::red));
    picker2->setTrackerMode(QwtPicker::ActiveOnly);
    connect(picker2, SIGNAL(moved(QwtDoublePoint)), 
            this, SLOT(markCurrent(QwtDoublePoint)));
    connect(picker2, SIGNAL(selected(QwtDoublePoint)), 
            this, SLOT(markCurrent(QwtDoublePoint)));

    picker3 = new QwtPlotPicker(QwtPlot::xBottom, QwtPlot::yLeft,
                                QwtPicker::PointSelection,
                                QwtPlotPicker::CrossRubberBand,
                                QwtPicker::AlwaysOn,
                                ui->profilePlot->canvas());
    picker3->setRubberBandPen(QColor(Qt::red));
    picker3->setTrackerMode(QwtPicker::ActiveOnly);
    connect(picker3, SIGNAL(moved(QwtDoublePoint)),
            this, SLOT(markCurrent(QwtDoublePoint)));
    connect(picker3, SIGNAL(selected(QwtDoublePoint)),
            this, SLOT(markCurrent(QwtDoublePoint)));

    pace6 = new QwtPlotMarker();
    pace6->setLineStyle(QwtPlotMarker::HLine);
    pace6->setYValue(10.0);
    pace6->setLinePen(QPen(Qt::red,1,Qt::DashLine));
    pace6->attach(ui->speedPlot);

    pace5 = new QwtPlotMarker();
    pace5->setLineStyle(QwtPlotMarker::HLine);
    pace5->setYValue(12.0);
    pace5->setLinePen(QPen(Qt::yellow,1,Qt::DashLine));
    pace5->attach(ui->speedPlot);

    pace4 = new QwtPlotMarker();
    pace4->setLineStyle(QwtPlotMarker::HLine);
    pace4->setYValue(15.0);
    pace4->setLinePen(QPen(Qt::green,1,Qt::DashLine));
    pace4->attach(ui->speedPlot);

    meanHR = new QwtPlotMarker();
    meanHR->setLineStyle(QwtPlotMarker::HLine);
    meanHR->setYValue(0.0);
    meanHR->setLinePen(QPen(Qt::green,1,Qt::DashLine));
    meanHR->attach(ui->pulsePlot);

    maxHR = new QwtPlotMarker();
    maxHR->setLineStyle(QwtPlotMarker::HLine);
    maxHR->setYValue(0.0);
    maxHR->setLinePen(QPen(Qt::red,1,Qt::DashLine));
    maxHR->attach(ui->pulsePlot);

    QwtText xaxis(tr("distance [m]"));
    QwtText vaxis(tr("<font color='blue'>speed [km/h]</font>"));
    QwtText paxis(tr("<font color='green'>pulse [bpm]</font>"));
    QwtText haxis(tr("<font color='black'>altitude [m]</font>"));
    ui->speedPlot->enableAxis(QwtPlot::xBottom, false);
    ui->speedPlot->setAxisTitle(QwtPlot::yLeft, vaxis);
    ui->pulsePlot->enableAxis(QwtPlot::xBottom, false);
    ui->pulsePlot->setAxisTitle(QwtPlot::yLeft, paxis);
    // ui->profilePlot->setTitleFont(
    ui->profilePlot->setAxisTitle(QwtPlot::xBottom, xaxis);
    ui->profilePlot->setAxisTitle(QwtPlot::yLeft, haxis);
    currentTrack->reset();
    ui->tabWidget->setCurrentIndex(0);
    createStatusBar();
}

MainWindow::~MainWindow()
{
    db.close();
    delete ui;
}

void MainWindow::on_actionStatistics_triggered()
{
    qDebug("on_actionStatistics_triggered()");
    StatisticsDialog *dlg = new StatisticsDialog();
    dlg->displayStats(ui->tableWidget);
    dlg->exec();
}

void MainWindow::on_actionQuit_triggered()
{
    close();
}

void MainWindow::displayMap()
{
    QList<Point*> points;
    QPointF pf;
    qreal dkm = 0.0, d;

    points.clear();
    int n = currentTrack->length(); 
    if (n < 3) return;

    QPen *cpen = new QPen();
    cpen->setColor(QColor(Qt::green));
    cpen->setWidth(3);
    pf = currentTrack->pointAt(0);
    // points << new CirclePoint(pf.x(), pf.y(), "start", Point::Middle, cpen);
    points << new ImagePoint(pf.x(), pf.y(),
                             ":/images/greenflag.png",
                             "start", Point::BottomRight);
    cpen->setColor(QColor(0,0,255,100)); // blue
    cpen->setWidth(2);
    for (int i = 1; i < n-1; i++) { 
        pf = currentTrack->pointAt(i);
        d = currentTrack->s[i];
        if (d-dkm > 1000.0) {
            points << new CirclePoint(pf.x(), pf.y(), "km", Point::Middle, cpen);
            dkm = d;
        } else {
            points << new Point(pf.x(), pf.y());
        }
    }
    cpen->setColor(QColor(Qt::red));
    cpen->setWidth(3);
    pf = currentTrack->pointAt(n-1);
    // points << new CirclePoint(pf.x(), pf.y(), "finish", Point::Middle, cpen);
    points << new ImagePoint(pf.x(), pf.y(),
                             ":/images/redflag.png",
                             "finish", Point::BottomLeft);
    mc->setView(currentTrack->centre());
    // currentTrack->dump();
    cpen->setColor(QColor(0,0,255,100)); // blue
    cpen->setWidth(3);

    if (lstr != NULL) {
        l->removeGeometry(lstr);
        delete(lstr);
    }
    lstr = new LineString(points, "", cpen);
    l->addGeometry(lstr);
}

void MainWindow::addZoomButtons()
{
    // create buttons as controls for zoom
    QPushButton* zoomin = new QPushButton("+");
    QPushButton* zoomout = new QPushButton("-");
    zoomin->setMaximumWidth(50);
    zoomout->setMaximumWidth(50);

    connect(zoomin, SIGNAL(clicked(bool)), mc, SLOT(zoomIn()));
    connect(zoomout, SIGNAL(clicked(bool)), mc, SLOT(zoomOut()));

    // add zoom buttons to the layout of the MapControl
    QVBoxLayout* innerlayout = new QVBoxLayout;
    innerlayout->addWidget(zoomin);
    innerlayout->addWidget(zoomout);
    mc->setLayout(innerlayout);
}

void MainWindow::trackSelected(int row)
{
    retrieveTrack(row+1);
}

void MainWindow::editTrack(int row, int column)
{
    bool ok;
    if (column != 1) return;

    QTableWidgetItem *item = ui->tableWidget->item(row, column);
    qDebug("clicked row=%d, column=%d", row, column);
    QString text = QInputDialog::getText(this, tr("edit track"),
					 tr("Enter new title:"), QLineEdit::Normal,
					 item->text(), &ok);
    if (ok && !text.isEmpty()) {
	QSqlQuery q;
	QString str = QString("UPDATE tracks SET title='%1' WHERE id=%2")
            .arg(text).arg(row+1);
	qDebug("UPDATE tracks SET title='%s' WHERE id=%d;", 
               text.toUtf8().constData(), row+1);
	q.exec(str);
	item->setText(text);
    }
}

void MainWindow::retrieveTrack(int id)
{
    QDateTime tstamp;
    qreal lon, lat, alt, d, hr;

    QString trackName = ui->tableWidget->item(id-1,1)->text();
    // qDebug() << trackName;
    currentTrack->setId(id);
    currentTrack->setName(trackName);

    QString str = QString("SELECT time, longitude, latitude, "
                          "altitude, distance, hrate "
                          "FROM gmn WHERE rid = %1 ORDER BY time").arg(id);
    // qDebug() << str;
    QSqlQuery q = QSqlQuery(str);
    currentTrack->reset();
    while (q.next()) {
        str = q.value(0).toString().replace('T',' ');
        tstamp = QDateTime::fromString(str, "yyyy-MM-dd hh:mm:ss");
        lon = q.value(1).toDouble();
        lat = q.value(2).toDouble();
        alt = q.value(3).toDouble();
        d   = q.value(4).toDouble();
        hr  = q.value(5).toDouble();
        currentTrack->addPoint(tstamp, lon, lat, alt, d, hr);
    }
    currentTrack->smoothVel2();
    showSpeed();
    showPulse();
    showProfile();
    displayMap();
    updateStatus();
}

void MainWindow::retrieveTable()
{
    QVariant v;
    QSqlQuery q = QSqlQuery("SELECT tracks.id, tracks.date, "
                            "tracks.title, max(gmn.distance) "
                            "FROM gmn,tracks "
                            "WHERE tracks.id=gmn.rid "
                            "GROUP by tracks.id,tracks.date,tracks.title "
                            "ORDER by id");
    while (q.next()) {
        v = q.value(0);
        int id = v.toInt();
        v = q.value(1);
        QString date = v.toString().replace('T',' ');
        v = q.value(2);
        QString trackName = v.toString();
        v = q.value(3);
        double d = v.toDouble();
        addToTable(id, trackName, date, d);
    }
}

void MainWindow::showSpeed()
{
    speed->setData(currentTrack->s, currentTrack->v);
    ui->speedPlot->setAxisScale(QwtPlot::xBottom, 0.0,
                                      currentTrack->s.last()+100.0);
    ui->speedPlot->setAxisAutoScale(QwtPlot::yLeft);
    ui->speedPlot->updateAxes();
    qreal ymax = ui->speedPlot->axisScaleDiv(QwtPlot::yLeft)->upperBound();
    ui->speedPlot->setAxisScale(QwtPlot::yLeft, 0.0, ymax);
    ui->speedPlot->axisScaleDraw(QwtPlot::yLeft)->setMinimumExtent(40);
    ui->speedPlot->replot();
}

void MainWindow::showPulse()
{
    qreal ymax = ui->speedPlot->axisScaleDiv(QwtPlot::xBottom)->upperBound();
    pulse->setData(currentTrack->s, currentTrack->p);
    ui->pulsePlot->setAxisScale(QwtPlot::xBottom, 0.0, ymax);
    ui->pulsePlot->axisScaleDraw(QwtPlot::yLeft)->setMinimumExtent(40);
    meanHR->setYValue(currentTrack->meanPulse());
    maxHR->setYValue(currentTrack->maxPulse());
    ui->pulsePlot->replot();
}

void MainWindow::showProfile()
{
    qreal ymax = ui->speedPlot->axisScaleDiv(QwtPlot::xBottom)->upperBound();
    profile->setData(currentTrack->s, currentTrack->h);
    ui->profilePlot->setAxisScale(QwtPlot::xBottom, 0.0, ymax);
    dateLabel->setText(currentTrack->date().toString());
    trackLabel->setText(currentTrack->name());
    /*
    QwtText title(currentTrack->date().toString()
                  + " - " + currentTrack->name());
    QFont titleFont("Helvetica", 12);    // , QFont::Bold);
    titleFont.setBold(false);
    title.setFont(titleFont);
    ui->profilePlot->setTitle(title);
    */
    ui->profilePlot->axisScaleDraw(QwtPlot::yLeft)->setMinimumExtent(40);
    ui->profilePlot->replot();
}

void MainWindow::markCurrent(QwtDoublePoint pos)
{
    QPen *cpen = new QPen(QColor(Qt::yellow));
    cpen->setWidth(3);
    qreal x;
    int i1, i2, i;

    x = pos.x();
    i1 = 0;
    i2 = currentTrack->length();
    // do a binary search for the closest index of position (x)
    while (i2-i1 > 1) {
        i = (i2+i1)/2;
        if (currentTrack->s[i] > x) {
            i2 = i;
        } else {
            i1 = i;
        }
    }

    QPointF pf = currentTrack->pointAt(i1);
    if (ipt == NULL) {
        ipt = new ImagePoint(pf.x(), pf.y(),
                             ":/images/runner.png",
                             "here", Point::BottomLeft);
        l->addGeometry(ipt);
    } else {
        ipt->setCoordinate(pf);
    }
    mc->updateRequestNew();
}

void MainWindow::on_actionOpen_triggered()
{
    QString program = "/usr/bin/garmin_dump";
    QString fileName;

    QSqlQuery q = QSqlQuery("SELECT max(id) FROM tracks");
    if (q.first()) {
        nextId = q.value(0).toInt()+1;
    } else {
        nextId = 0;
        qDebug() << "failed to get new track id";
        return;
    }

    qDebug() << "next id " << nextId;

    QSettings settings("qtrainer", "qtrainer");
    QString td = settings.value("garmin/directory", QDir::homePath()).toString();
    if (lastdir.isEmpty()) lastdir = td;

    qDebug() << "look for gmn files in:" << lastdir;
    fileName = QFileDialog::getOpenFileName(this, tr("Add Track"), lastdir,
                                            tr("Tracks (*.gmn)"));
    if (fileName.isNull()) return;

    QFileInfo fi(fileName);
    lastdir = fi.absolutePath();

    QFile file(fileName);
    if (!file.open(QIODevice::ReadOnly)) {
        QMessageBox::warning(this, tr("QTrainer"),
                             tr("Failed to open file %1:\n%2.")
                             .arg(file.fileName())
                             .arg(file.errorString()));
        return;
    }
    QStringList arguments;
    arguments << fileName;
    qDebug() << arguments;
    xml.clear();
    noTrackPoints = 0;
    xml.addData("<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n");
    xml.addData("<garmin>\n");
    proc = new QProcess();
    connect(proc, SIGNAL(readyReadStandardOutput()), 
            this, SLOT(readPipe()));
    connect(proc, SIGNAL(finished(int,QProcess::ExitStatus)), 
            this, SLOT(donePipe(int,QProcess::ExitStatus)));
    proc->start(program, arguments);
}

void MainWindow::readPipe()
{
    QByteArray bytes = proc->readAllStandardOutput();
    noTrackPoints += bytes.size();
    // qDebug() << "more data" << noTrackPoints/130;
    xml.addData(bytes);
}

void MainWindow::donePipe(int exitCode, QProcess::ExitStatus exitStatus)
{
    if (exitStatus == QProcess::NormalExit) {
        xml.addData("</garmin>");
        /* each point element is roughly 130 characters long */
        noTrackPoints /= 130;
        parseXML();
    } else {
        qDebug() << "crashed";
        qDebug() << exitCode;
    }
}

void MainWindow::parseXML()
{
    QXmlStreamAttributes attr;
    double lon, lat, alt, d, hrate, total;
    QString str, start, title;
    bool Forerunner305, ok;
    int type;
    int iPoints = 0;
    QSqlQuery q = QSqlQuery("INSERT INTO gmn "
                "(rid,time,latitude,longitude,altitude,distance,hrate) "
                            "VALUES (:id,:time,:lat,:lon,:alt,:dist,:hr)");

    title = QString("<untitled>");
    qDebug() << "filling db for track " << nextId;
    qDebug() << "expecting" << noTrackPoints << "points";
    QXmlStreamReader::TokenType token;
    total = 0.0;
    QProgressDialog progress("Reading track...", QString(), 
                             0, noTrackPoints);
    progress.setMinimumDuration(1000);

    db.transaction();
    while (!xml.atEnd()) {
        token = xml.readNext();
        switch (token) {
          case QXmlStreamReader::StartElement:
            if (xml.name() == "lap") {
                attr = xml.attributes();
                start = attr.value("start").toString();
                start.truncate(19);
            } else if (xml.name() == "point") {
                attr = xml.attributes();
                str = attr.value("time").toString();
                str.truncate(19);
                // str = str.replace('T',' ');
                type = attr.value("type").toString().toInt();
                if (type == 304 && attr.hasAttribute("lat")) {
                    q.bindValue(":id", nextId);
                    q.bindValue(":time", str);
                    lat = attr.value("lat").toString().toDouble();
                    q.bindValue(":lat", lat);
                    lon = attr.value("lon").toString().toDouble();
                    q.bindValue(":lon", lon);
                    alt = attr.value("alt").toString().toDouble();
                    q.bindValue(":alt", alt);
                    d = attr.value("distance").toString().toDouble();
                    if (d > total) total = d;
                    q.bindValue(":dist", d);
                    Forerunner305 = (attr.hasAttribute("hr"));
                    if (Forerunner305) {
                        hrate = attr.value("hr").toString().toDouble();
                        q.bindValue(":hr", hrate);
                    } else {
                        hrate = 0.0;
                        q.bindValue(":hr", QVariant(QVariant::Double));
                    }
                    ok = q.exec();
                    QApplication::processEvents();
                    if (!ok) {
                        qDebug() << q.lastError();
                    } else {
                        iPoints++;
                        progress.setValue(iPoints);
                    }
                }
            }
            break;
          case QXmlStreamReader::EndElement:
            break;
          default:
            break;
        }
    }
    if (xml.hasError()) {
        qDebug() << "xml error at line " << xml.lineNumber() 
                 << ", column " << xml.columnNumber() 
                 << "(" << xml.characterOffset() << ")";
        qDebug() << xml.errorString();
        db.rollback();
    } else {
        if (nextId > 0 && iPoints > 0) {
            addToDb(nextId, title, start);
            db.commit();
            start = start.replace('T',' ');
            addToTable(nextId, title, start, total);
            qDebug() << "added" << iPoints << "points";
            progress.setValue(noTrackPoints);
        } else {
            qDebug() << "no records found";
            db.rollback();
        }
    }
    return;
}

void MainWindow::addToDb(int id, const QString &title, const QString &start)
{
    QString str = QString("INSERT INTO tracks (id, date, title) "
                           "VALUES (%1,'%2','%3')").arg(id).arg(start).arg(title);
    QSqlQuery q = QSqlQuery(str);
    q.exec();
    qDebug() << str;
}

void MainWindow::addToTable(int id, const QString &title, 
                            const QString &start, double total)
{
    QTableWidgetItem *item;
    if (ui->tableWidget->rowCount() < id) {
        ui->tableWidget->setRowCount(id);
    }

    item = new QTableWidgetItem(start);
    item->setFlags(Qt::ItemIsSelectable|Qt::ItemIsEnabled);
    ui->tableWidget->setItem(id-1, 0, item);
    item = new QTableWidgetItem(title);
    item->setFlags(Qt::ItemIsSelectable|Qt::ItemIsEnabled);
    ui->tableWidget->setItem(id-1, 1, item);
    item = new QTableWidgetItem(QString().setNum(total/1000.0,'f',3));
    item->setFlags(Qt::ItemIsSelectable|Qt::ItemIsEnabled);
    item->setTextAlignment(Qt::AlignRight|Qt::AlignVCenter);
    ui->tableWidget->setItem(id-1, 2, item);
}

void MainWindow::on_actionAbout_triggered()
{
    QMessageBox::about(this, tr("About QTrainer"),
                       tr("<h2>QTrainer "APP_VERSION" </h2>"
                          "<p>Copyright &copy; 2011 Michael Olberg"
                          "<p>QTrainer is a small application to display "
                          "tracks recorded by your Garmin Forerunner 205 "
                          "or 305 device."
                          "<p>It needs the garmin-forerunner-tools installed "
                          "in order to work fully, else some functionality "
                          "will be disabled."));
}

void MainWindow::createStatusBar()
{
    // QFrame *empty = new QFrame(this);
    // empty->setWidth
    dateLabel = new QLabel(tr("                        "), this);
    dateLabel->setMinimumSize(dateLabel->sizeHint());
    statusBar()->addWidget(dateLabel);
    // statusBar()->addWidget(empty);
    trackLabel = new QLabel(this);
    trackLabel->setAlignment(Qt::AlignHCenter);
    // trackLabel->setMinimumSize(trackLabel->sizeHint());
    statusBar()->addWidget(trackLabel, 1);
    // statusBar()->addWidget(empty);
    statistics = new QLabel(tr("                       "), this);
    statistics->setMinimumSize(statistics->sizeHint());
    statusBar()->addWidget(statistics, 4);
}

void MainWindow::updateStatus()
{
    double vmean = currentTrack->meanPace();
    double vmax  = currentTrack->maxPace();
    double pmean = currentTrack->meanPulse();
    double pmax  = currentTrack->maxPulse();
    double dist  = currentTrack->totalLength();
    QString duration = currentTrack->totalTime().toString();
    QString stat = QString("duration = %1, distance = %2 km, "
                           "mean/max pace = %3/%4 min/km, "
                           "mean/max pulse = %5/%6 bpm")
        .arg(duration).arg(dist,0,'f',3)
        .arg(vmean,0,'f',2).arg(vmax,0,'f',2)
        .arg(pmean,0,'f',0).arg(pmax,0,'f',0);
    statistics->setText(stat);
}

void MainWindow::on_actionImport_triggered()
{
    GarminDialog *dlg = new GarminDialog();
    dlg->exec();
}

bool MainWindow::checkDatabase()
{
    bool ok = true, hasTracks = false, hasGmn = false;

    QSqlQuery q = QSqlQuery("SELECT name FROM sqlite_master "
                            "WHERE type='table'");
    q.exec();
    while (q.next()) {
         QString name = q.value(0).toString();
         if (name.compare("tracks", Qt::CaseInsensitive) == 0) hasTracks = true;
         if (name.compare("gmn",    Qt::CaseInsensitive) == 0) hasGmn = true;
    }
    if (!hasTracks) {
        qDebug("database table 'tracks' does not exist");
        ok = q.exec("CREATE TABLE tracks ("
                    "id INTEGER PRIMARY KEY, "
                    "date timestamp, "
                    "title varchar(64))");
    }
    if (!ok) return ok;

    if (!hasGmn) {
        qDebug("database table 'gmn' does not exist");
        ok = q.exec("CREATE TABLE gmn ("
                    "time timestamp, "
                    "latitude float, "
                    "longitude float, "
                    "altitude float, "
                    "distance float, "
                    "hrate float)");
    }
    return ok;
}
