#include <QDebug>

#include <stdlib.h>
#include <math.h>

#include "track.h"
#include "savgol.h"

TrackPoint::TrackPoint()
{
    tstamp = QDateTime();
    longitude = 0.0;
    latitude = 0.0;
    altitude = 0.0;
    distance = 0.0;
    hrate = 0.0;
}

TrackPoint::TrackPoint(const QDateTime time, const double lon, 
                       const double lat, const double alt, 
                       const double d, const double hr)
{
    tstamp = time;
    longitude = lon;
    latitude = lat;
    altitude = alt;
    distance = d;
    hrate = hr;
}

QPointF TrackPoint::position() const
{
    return QPointF(longitude,latitude);
}

QString TrackPoint::toString() const
{
    QString str = tstamp.toString();
    str.append(" ").append(QString().setNum(longitude,'f',4));
    str.append(" ").append(QString().setNum(latitude,'f',4));
    str.append(" ").append(QString().setNum(altitude,'f',4));
    str.append(" ").append(QString().setNum(distance/1000.0,'f',3));
    str.append(" ").append(QString().setNum(hrate,'f',1));
    return str;
}

void TrackPoint::dump() const
{
    qDebug() << toString();
}

Track::Track(QPointF home, int id)
{
    if (id == -1) {
        when = QDateTime::currentDateTime();
        which = "unknown";
        where = home;
    }
}

QPointF Track::centre()
{
    double lon_m = 0.0;
    double lat_m = 0.0;

    int n = length();
    if ((_id != -1) && (n > 0)) {
        for (int i = 0; i < n; i++) {
            lon_m += pos[i].x();
            lat_m += pos[i].y();
        }
        lon_m /= (double)n;
        lat_m /= (double)n;
        where.setX(lon_m);
        where.setY(lat_m);
    } else {
        // Kabelgatan, Kungsbacka
        lon_m = 12.070792;
        lat_m = 57.503674;
        // Stockholm Stadium
        // lon_m = 18.079033;
        // lat_m = 59.345303;
    }
    return where;
    // return QPointF(lon_m, lat_m);
}

QString Track::name() const
{
    return which;
}

void Track::setName(QString name)
{
    which = name;
}

int Track::id()
{
    return _id;
}

void Track::setId(int id)
{
    _id = id;
}

QDateTime Track::date() const
{
    return when;
}

void Track::addPoint(const QDateTime tstamp, const double lon, 
                     const double lat, const double alt, 
                     const double d, const double hr)
{
    qreal dt, u;
    int n;

    h.append(alt);
    s.append(d);
    p.append(hr/1.0);
    pos.append(QPointF(lon, lat));
    if (!filled) {
        when = tstamp;
        t.append(0.0);
        v.append(0.0);
        filled = true;
    } else {
        dt = (qreal)when.secsTo(tstamp);
        t.append(dt);
        v.append(0.0);
        n = t.size();
        if (n > 2) {
            u = 3.6*(s[n-1]-s[n-3])/(t[n-1]-t[n-3]);
            v[n-2] = u;
        }
    }
}

void Track::reset()
{
    t.clear();
    h.clear();
    s.clear();
    v.clear();
    p.clear();
    pos.clear();
    // gps.clear();
    filled = false;
}

void Track::dump() const
{
    TrackPoint tp;

    int n = t.size();
    qDebug() << "track has " << n << "trackpoints";
    for (int i = 0; i < n; i++) {
        QString str = when.addSecs((int)t[i]).toString();
        qreal longitude = pos[i].x();
        qreal latitude = pos[i].y();
        str.append(" ").append(QString().setNum(longitude,'f',4));
        str.append(" ").append(QString().setNum(latitude,'f',4));
        str.append(" ").append(QString().setNum(h[i],'f',4));
        str.append(" ").append(QString().setNum(s[i]/1000.0,'f',3));
        str.append(" ").append(QString().setNum(p[i]*10.0,'f',1));

        qDebug() << str;
    }
}

int Track::length() const
{
    return t.size();
}

QPointF Track::pointAt(int i) const
{
    return pos[i];
}

double Track::mean(QVector<qreal> x)
{
    double _mean = 0.0;

    int n = x.size();
    for (int i = 0; i < n; i++) {
        _mean += x[i];
    }
    _mean /= (double)n;
    return _mean;
}

double Track::max(QVector<qreal> x)
{
    double _max = 0.0;

    int n = x.size();
    _max = x[0];
    for (int i = 1; i < n; i++) {
        if (x[i] > _max) _max = x[i];
    }
    return _max;
}

qreal Track::median(qreal a, qreal b, qreal c)
{
    if (a < b) {
        if (c < a)     return a;
        else {
            if (c < b) return c;
            else       return b;
        }
    } else {
        if (c < b)     return b;
        else {
            if (c < a) return c;
            else       return a;
        }
    }
}


void Track::smoothVel0()
{
    int n = v.size();
    qreal med0, med1;

    med0 = median(v[0], v[1], v[2]);
    for (int i=2; i < n-1; i++) {
        med1 = median(v[i-1], v[i], v[i+1]);
        v[i-1] = med0;
        med0 = med1;
    }
    v[n-2] = med0;
}

void Track::smoothVel1()
{
    int i, j;
    const int m=7;
    qreal a[m], mean;

    int n = v.size();
    if (n < m) return;
    for (i = 0; i < m; i++) a[i] = v[i];

    for (i = (m-1)/2; i < n-(m-1)/2-1; i++) {
        mean = 0.0;
        for (j = 0; j < m; j++) mean += a[j];
        mean /= m;
        v[i] = mean;
        for (j = 1; j < m; j++) a[j-1] = a[j];
        a[m-1] = v[i+(m-1)/2+1];
    }
    mean = 0.0;
    for (j = 0; j < m; j++) mean += a[j];
    mean /= m;
    v[n-(m-1)/2] = mean;
}

void Track::smoothVel2()
{
    int n = 5;
    int order = 4;

    SavGolFilter sg(n, n, 0, order);
    qreal *b = sg.filter(v);
    for (int i = 0; i < v.size(); i++) {
        v[i] = b[i];
    }
}

double Track::meanPace() const
{
    return (t.last()/60.0)/(s.last()/1000.0);
}

double Track::maxPace() const
{
    return 60.0/max(v);
}

double Track::meanPulse() const
{
    return mean(p);
}

double Track::maxPulse() const
{
    return max(p);
}

double Track::totalLength() const
{
    return s.last()/1000.0;
}

QTime Track::totalTime() const
{
    int secs = (int)floor(t.last());
    int hours = secs/3600;
    secs -= hours*3600;
    int minutes = secs/60;
    secs -= minutes*60;

    return QTime(hours, minutes, secs);
}
