/**************************** XS ********************************************
Copyright (C) 2000-2023  P. Bergman

This program is free software; you can redistribute it and/or modify it under 
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 2 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
*****************************************************************************/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <Xm/Xm.h>
#include <Xm/PushBG.h>
#include <Xm/RowColumn.h>
#include <Xm/Frame.h>
#include <Xm/Label.h>
#include <Xm/Text.h>

#include "defines.h"
#include "global_structs.h"
#include "menus.h"
#include "dialogs.h"

/*** Global variables ***/
extern MOD    mods[MAXMOD];
extern int    nmod, rbox_sel, mod_sel;
extern VIEW  *vP; 

void PostErrorDialog(Widget, char *);
void PostWarningDialog(Widget, char *);
void PostMessageDialog(Widget, char *);
void ManageDialogCenteredOnPointer(Widget);
Widget CreateOptionMenu(Widget, MenuBarItem *);
void SetDefaultOptionMenuItem(Widget, int);

void w_printf(Widget, char *, ...);
void send_line(char *);
void wdscanf(Widget, double *);
void wiscanf(Widget, int *);
void UpdateData(int, int);

int        count_scans(DataSetPtr);
list       scan_iterator(list, DataSetPtr);
scanPtr    copy_scan(DataSetPtr, scanPtr);
list      *get_listlist();
DataSetPtr new_dataset(list *, char *, DataSetPtr);
void       DeleteLastDataSet();

/*** Local variables ***/
typedef struct {
    int    type, nskip;
    double dsig_side, dsig_peak;
    Widget lside, lpeak, lskip;
} despike;

static despike Despike;
static string buf;

static void DespikeTypeCallback(Widget, char *, XmAnyCallbackStruct *);

MenuItem DespikeMenuData[] = {
  {"Single channel detection", &xmPushButtonGadgetClass,
   XK_VoidSymbol, NULL, NULL, False, NULL, DespikeTypeCallback, "0", NULL},
  {"Single channel detection and edge channel modification", &xmPushButtonGadgetClass,
   XK_VoidSymbol, NULL, NULL, False, NULL, DespikeTypeCallback, "1", NULL},
EOI};

MenuBarItem DespikeOptionMenu = {
   "Despike method", ' ', True, DespikeMenuData
};

static char *Despike_Help = "\
                           Despike help\n\
                           ------------\n\
In this dialog you can select different ways of despiking spectra.\n\
Use with great care as narrow real lines can be removed. The options are:\n\
    Single channel detection:\n\
          Detect single channel spikes using three channel box.\n\
	  Spikes can be positive or negative.\n\
    Single channel + edge detection:\n\
          Same as above but change also outermost edge channels.\n\
";

void init_despike_parameters()
{
    Despike.type = 0;
    Despike.nskip = 3;
    Despike.dsig_side  = 2.5;
    Despike.dsig_peak  = 5.5;
}

void new_mod(int chan, double new_value)
{
    string buf;
    
    void UpdateData(int, int);

    if (!vP->s) {
        PostErrorDialog(NULL, "No data at all to modify.");
        return;
    }

    if (nmod < MAXMOD) {
        if (vP->s && chan >= 0 && chan < vP->s->nChan) {
            mods[nmod].chan = chan;
            mods[nmod].old  = vP->s->d[chan];
            mods[nmod].new  = new_value;
            nmod++;
            vP->s->d[chan] = new_value;
            vP->s->saved = 0;
            UpdateData(SCALE_NONE, REDRAW);
        } else {
            sprintf(buf, "Selected channel %d is outside of spectrum.", chan);
            PostErrorDialog(NULL, buf);
        }
    } else {
        sprintf(buf, "Too many channel modifications: %d > %d.",
                nmod+1, MAXMOD);
        PostErrorDialog(NULL, buf);
    }
}

void mod_reset(Widget w, char *client_data, XtPointer call_data)
{
    int n;
    
    void draw_main();

    if (!vP->s) {
        PostErrorDialog(NULL, "No data at all to modify.");
        return;
    }

    if (strncmp(client_data, "all", 3) == 0) {
        for (n=nmod-1; n>=0; n--) {
            if (mods[n].chan >= 0 && mods[n].chan < vP->s->nChan)
                vP->s->d[mods[n].chan] = mods[n].old;
        }
        vP->s->saved = 0;
        nmod = 0;
    } else {     
        nmod--;
        if (nmod < 0) {
            nmod = 0;
        } else {
            if (mods[nmod].chan >= 0 && mods[nmod].chan < vP->s->nChan)
                vP->s->d[mods[nmod].chan] = mods[nmod].old;
            vP->s->saved = 0;
        }
    }
    draw_main();
}

void channel_mod(Widget w, char *client_data, XtPointer call_data)
{
    if (mod_sel == 0) {
        rbox_sel = 0;
        mod_sel = 1;
    }
}

/*
static void make_dheap( double x[ ], int n )
{
    int i, s, f ;
    double val ;
    for ( i = 1 ; i < n ; i++ )
    {
        val = x[i] ;
        s = i ;
        f = ( s - 1 ) / 2 ;
        while ( s > 0 && x[f] < val )
        {
            x[s] = x[f] ;
            s = f ;
            f = ( s - 1 ) / 2 ;
        }
        x[s] = val ;
    }
}

static void heap_dsort( double x[ ], int n )
{
    int i, s, f ;
    double ivalue ;
    for ( i = n - 1 ; i > 0 ; i-- )
    {
        ivalue = x[i] ;
        x[i] = x[0] ;
        f = 0 ;

        if ( i == 1 )
            s = -1 ;
        else
            s = 1 ;

        if ( i > 2 && x[2] > x[1] )
            s = 2 ;

        while ( s >= 0 && ivalue < x[s] )
        {
            x[f] = x[s] ;
            f = s ;
            s = 2 * f + 1 ;

            if ( s + 1 <= i - 1 && x[s] < x[s + 1] )
                s++ ;
            if ( s > i - 1 )
                s = -1 ;
        }
        x[f] = ivalue ;
    }
}

static void sortdoublevector(double v[], int n)
{
    make_dheap(v, n);
    heap_dsort(v, n);
}
   */

/* 
#define DEBUG
 */

static int d_cmp_func(const void *a, const void *b)
{
    double *x = (double *) a;
    double *y = (double *) b;

    if (*x < *y)
      return -1;
    else if (*x > *y)
      return 1;
      
    return 0;
}

static scanPtr DespikeSpe(DataSetPtr d, scanPtr s, despike *h)
{
    int n, nskip, nused, nspike, nch, has_nan, nan_ch=-1;
    double s1, s2, ave, sig, med, a, dy, dy1, dy2;
    scanPtr new = NULL;
    
    if (!d || !s || !h) return new;
    
    new = copy_scan(d, s);
    if (!new) return new;
    
    /* sortdoublevector(new->d, new->nChan); */ 
/* Use unix qsort, need stdlib.h */
    qsort(new->d, new->nChan, sizeof(double), d_cmp_func);
    
    med = new->d[new->nChan/2];
    
    nskip = h->nskip;
    if (nskip < 0) nskip=0;
    
    nused=0;
    s1 = s2 = 0.0;
    for (n=nskip; n<new->nChan-nskip; n++) {
      a = new->d[n];
/* only use proper data */
      if (isfinite(a)) {
        s1 += a;
        s2 += a*a;
        nused++;
      }
    }
    
    ave = sig = 0.0;
    if (nused > 1) {
      ave = s1/(double)nused;
      sig = sqrt((s2 - s1*s1/(double)nused)/(double)(nused-1));
    }

#ifdef DEBUG    
    printf("%s nCh=%d ave=%f sig=%f med=%f\n", new->name, new->nChan, ave, sig, med);
    printf(" type=%d side=%.1f peak=%.1f\n", h->type, h->dsig_side, h->dsig_peak);
#endif
    
    has_nan = 0;
    for (n=0; n<new->nChan; n++) {
      a = s->d[n];
/* using the math.h isfinite() here */
      if (isfinite(a)) {
        new->d[n] = a;
      } else { /* either nan or inf here */
        new->d[n] = med;
	nan_ch = n;
	has_nan++;
      }
    }
    
    if (h->type == 1) {
      for (n=0; n<nskip; n++) {
        if (n < new->nChan - nskip) new->d[n] = med;
        if (new->nChan-1-n > nskip) new->d[new->nChan-1-n] = med;
      }
    }
    
    nspike = 0;
    for (n=1; n<new->nChan-1; n++) {
      dy1 = new->d[n] - new->d[n-1];
      dy2 = new->d[n+1] - new->d[n];
      if (dy1 * dy2 >= 0.0) continue;
      a = (new->d[n+1] + new->d[n-1])/2.;
      dy = (new->d[n+1] - new->d[n-1]);
      if (fabs(dy) > h->dsig_side*sig || fabs(a-ave) > h->dsig_side*sig) continue;
      if (fabs(new->d[n] - a) < h->dsig_peak*sig) continue;
#ifdef DEBUG    
      printf("n=%d side/sig=%f,%f peak/sig=%f\n", n,
             fabs(dy)/sig, fabs(a-ave)/sig,fabs(new->d[n] - a)/sig);
#endif
      new->d[n] = a;
      nch = n;
      nspike++;
    }
    
    new->nspike = nspike;   
    if (nspike==0)
      sprintf(buf, "%s: detected no spikes", new->name);
    else if (nspike==1)
      sprintf(buf, "%s: removed a single spike at channel %d", new->name, nch);
    else if (nspike < 4)
      sprintf(buf, "%s: removed %d single channel spikes", new->name, nspike);
    else
      sprintf(buf, "WARNING %s: removed %d single channel spikes", new->name, nspike);
    send_line(buf);
    
    if (has_nan) {
      if (has_nan == 1)
        sprintf(buf, "WARNING %s had a NaN/inf value at channel %d, set to median value %f",
                new->name, nan_ch, med);
      else
        sprintf(buf, "WARNING %s had %d NaN/inf values, set to median value %f",
                new->name, has_nan, med);
      send_line(buf);
    }
    
    return new;
}

static void DoDespike(Widget w, StdForm *sf, XmAnyCallbackStruct *cb)
{
    int n;
    list curr = NULL;
    scanPtr new;
    DataSetPtr dsp;
    despike *s = (despike *)sf->any;

    wdscanf(sf->edit[1], &(s->dsig_side));
    wdscanf(sf->edit[3], &(s->dsig_peak));
    wiscanf(sf->edit[5], &(s->nskip));

    dsp = new_dataset(get_listlist(), "Despiked", vP->from);
    if (!dsp) {
        PostErrorDialog(w, "Out of memory when allocating dataset.");
        return;
    }

    if (s->type == 0 && count_scans(vP->from) > 0) {
        n = 0;
        while ( (curr = scan_iterator(curr, vP->from)) != NULL) {
            new = DespikeSpe(dsp, (scanPtr)DATA(curr), s);
            if (!new) break;
            n++;
        }
        if (n == 0) {
            PostErrorDialog(w, "Out of memory when allocating scan.");
            DeleteLastDataSet();
            return;
        }
        vP->to = vP->from = dsp;
        vP->s = (scanPtr)DATA(dsp->scanlist);
        sprintf(buf, "Despiked %d spectra.", n);
    } else if (s->type == 1 && count_scans(vP->from) > 0) {
        n = 0;
        while ( (curr = scan_iterator(curr, vP->from)) != NULL) {
            new = DespikeSpe(dsp, (scanPtr)DATA(curr), s);
            if (!new) break;
            n++;
        }
        if (n == 0) {
            PostErrorDialog(w, "Out of memory when allocating scan.");
            DeleteLastDataSet();
            return;
        }
        vP->to = vP->from = dsp;
        vP->s = (scanPtr)DATA(dsp->scanlist);
        sprintf(buf, "Despiked %d spectra.", n);
    } else {
        PostErrorDialog(w, "Couldn't find any spectra to despike.");
        return;
    }

    send_line(buf);
    
    if (s->type == 0) {
        sprintf(dsp->name, "Despiked %s 0 [%.1f,%.1f]",
                vP->s->name, s->dsig_side, s->dsig_peak);
    } else if (s->type == 1) {
        sprintf(dsp->name, "Despiked %s 1 [%.1f,%.1f,%d]",
                vP->s->name, s->dsig_side, s->dsig_peak, s->nskip);
    }

    UpdateData(SCALE_ONLY_X, REDRAW);
}

static void SetDespikeLabelStrings(StdForm *sf)
{
    Widget val, ref, nsk;
    
    if (sf) {
        val = sf->edit[0];
        ref = sf->edit[2];
        nsk = sf->edit[4];
    } else {
        val = Despike.lside;
        ref = Despike.lpeak;
        nsk = Despike.lskip;
    }
    
    if (Despike.type == 0) {
        w_printf(val, "%s", "Side sigma multiple");
        w_printf(ref, "%s", "Peak sigma multiple");
        w_printf(nsk, "%s", "Only used in median calculation");
    } else if (Despike.type == 1) {
        w_printf(val, "%s", "Side sigma multiple");
        w_printf(ref, "%s", "Peak sigma multiple");
        w_printf(nsk, "%s", "No of edge channels to set on each end");
    }
}

static void DespikeTypeCallback(Widget w, char *str, XmAnyCallbackStruct *cb)
{
    int type = atoi(str);
    
    if (type != Despike.type) {
        Despike.type = atoi(str);
        SetDespikeLabelStrings(NULL);
    }
}

void PostDespikeDialog(Widget wid, char *cmd, XtPointer call_data)
{
    Widget rc, rc2, fr, menu;
    Widget w = wid;
    StdForm *sf;

    while (!XtIsWMShell(w))
        w = XtParent(w);

    sf = PostStdFormDialog(w, "Despike spectra",
             BUTT_APPLY, (XtCallbackProc)DoDespike, NULL,
             BUTT_CANCEL, NULL, NULL,
             BUTT_HELP, NULL, Despike_Help,
	     6, NULL);
    sf->any = (XtPointer)(&Despike);

    rc = XtVaCreateManagedWidget("scalerc", xmRowColumnWidgetClass, sf->form,
                                 XmNorientation, XmVERTICAL,
                                 NULL);
                                  
    menu = CreateOptionMenu(rc, &DespikeOptionMenu);
    SetDefaultOptionMenuItem(menu, Despike.type);
				                   
    fr = XtVaCreateManagedWidget("frame", xmFrameWidgetClass, rc, 
				 XmNshadowType, XmSHADOW_OUT, NULL);

    rc2 = XtVaCreateManagedWidget("rowcol", xmRowColumnWidgetClass, fr,
                                  XmNorientation, XmVERTICAL,
				  NULL);
    
    sf->edit[0] = XtCreateManagedWidget("label", xmLabelWidgetClass,
                                        rc2, NULL, 0);
    sf->edit[1] = XtCreateManagedWidget("edit", xmTextWidgetClass,
                                        rc2, NULL, 0);
    sf->edit[2] = XtCreateManagedWidget("label", xmLabelWidgetClass,
                                        rc2, NULL, 0);
    sf->edit[3] = XtCreateManagedWidget("edit", xmTextWidgetClass,
                                        rc2, NULL, 0);
    sf->edit[4] = XtCreateManagedWidget("label", xmLabelWidgetClass,
                                        rc2, NULL, 0);
    sf->edit[5] = XtCreateManagedWidget("edit", xmTextWidgetClass,
                                        rc2, NULL, 0);
    
    Despike.lside = sf->edit[0];
    Despike.lpeak = sf->edit[2];
    Despike.lskip = sf->edit[4];
    
    ArrangeStdFormDialog(sf, rc);

    SetDespikeLabelStrings(sf);
    
    w_printf(sf->edit[1], "%f", Despike.dsig_side);
    w_printf(sf->edit[3], "%f", Despike.dsig_peak);
    w_printf(sf->edit[5], "%d", Despike.nskip);
    
    XtManageChild(menu);
    
    ManageDialogCenteredOnPointer(sf->form);
}
